var AbstractFormatter;
var AbstractHandler;
var Configurator;

var assert = require('assert');
var util = require('util');
var _ = require('lodash');

/**
 * Log message
 * @class LogEntry
 */
var LogEntry = require('inherit')({
    /**
     * @param {AbstractFormatter} format
     * @param {AbstractHandler} handler
     * @constructor
     * @class LogEntry
     */
    __constructor: function(format, handler) {
        AbstractFormatter = AbstractFormatter || require('./formatter/AbstractFormatter');
        AbstractHandler = AbstractHandler || require('./handler/AbstractHandler');
        assert(format instanceof AbstractFormatter, 'Formatter should be a LogFormatter');
        assert(handler instanceof AbstractHandler, 'Handler should be a LogHandler');

        this._formatter = format;
        this._handler = handler;

        this._values = {
            level: null,
            type: null,
            logId: null,
            message: null,
            timestamp: null
        };

        this._written = false;
    },

    /**
     * Set a level for the entry
     * @param {string} level
     */
    level: function(level) {
        Configurator = Configurator || require('./Configurator');
        var knownLevels = Configurator.knownLevels;

        assert(knownLevels.indexOf(level) > -1, 'Level should be one of: ' + knownLevels.join(', '));
        this._values.level = level;
        return this;
    },

    /**
     * Set a type for the entry
     */
    type: function() {
        var type = Array.prototype.slice.call(arguments);

        type.forEach(function(type) {
            assert(type && typeof type === 'string', 'Type should be a string');
        });

        this._values.type = (this._values.type || []).concat(type);
        return this;
    },

    /**
     * Set a log id common for the entries of the same request
     * @param {string} logId
     */
    logId: function(logId) {
        assert(logId && typeof logId === 'string', 'Log ID should be a string');
        this._values.logId = logId;
        return this;
    },

    /**
     * Finally write the message
     */
    write: function() {
        assert(this._written === false, 'Message has already been written once');
        assert(this._values.level !== null, 'Level should be specified');

        this._values.message = util.format.apply(
            null,
            _.map(arguments, function(arg) {
                if (arg instanceof Error) {
                    return arg.stack || (typeof arg.toString === 'function' && arg.toString()) || arg;
                }

                return arg;
            })
        );

        this._values.timestamp = Date.now();
        this._handler.handle(this);
        this._written = true;
    },

    getFormatted: function() {
        return this._formatter.format(this);
    },

    getFormatter: function() {
        return this._formatter;
    },

    getHandler: function() {
        return this._handler;
    },

    getType: function() {
        return this._values.type || ['unknown'];
    }
});

//Setting the simple getters
['level', 'logId', 'message', 'timestamp'].forEach(function(field) {
    LogEntry.prototype['get' + field.charAt(0).toUpperCase() + field.slice(1)] = function() {
        return this._values[field] || 'unknown';
    };
});

module.exports = LogEntry;
