var Configurator = require('./Configurator');
var LogEntry = require('./Entry');

/**
 * A logger interface
 * @class Logger
 *
 * @example Default logger
 *  require('plog')
 *      .warn()
 *      .type('project', 'class', 'method')
 *      .logId(id)
 *      .write('message');
 *
 * @example Remembering logId
 *  var Logger = require('plog');
 *  var logger = new Logger(logId, 'project', 'class');
 *  logger
 *      .warn()
 *      .type('method')
 *      .write('message'); //Would write the same thing as a call above
 *
 * @example require('plog')
 *      .configure()
 *      .minLevel(level);
 */
var Logger = require('inherit')(
    {
        __constructor: function(logId) {
            this._logId = logId;
            this._type = Array.prototype.slice.call(arguments, 1);
        },

        type: function() {
            this._type = this._type.concat(Array.prototype.slice.call(arguments));
        },

        getType: function() {
            return this._type;
        },

        getLogId: function() {
            return this._logId;
        }
    },
    {
        configure: Configurator.instance,
        log: function() {
            var config = Configurator.instance();

            return new LogEntry(config.getFormatter(), config.getHandler());
        }
    }
);

Configurator.knownLevels.forEach(function(level) {
    Logger[level] = function() {
        return Logger.log().level(level);
    };

    Logger.prototype[level] = function() {
        var entry = Logger[level]();

        entry.type.apply(entry, this._type);

        if (this.getLogId()) {
            entry.logId(this.getLogId());
        }

        if (arguments.length > 0) {
            entry.write.apply(entry, arguments);
        }

        return entry;
    };
});

module.exports = Logger;
