const got = require('got');
const PLog = require('plog');
const urlParse = require('url').parse;
const _ = require('lodash');
const tools = require('./tools');

const DEFAULT_TIMEOUT = 10000;

module.exports = ({url, params = {}, logID, logSuffix, hooks = null}) => {
    const {method = 'get', body, form, responseType = 'json', timeout = DEFAULT_TIMEOUT, ...requestParams} = params;
    const logInput = tools.clearLog(method === 'get' ? urlParse(url, true).query : body ? body : form ? form : {});
    const logPathname = urlParse(url).pathname || '';
    const logger = new PLog(
        logID,
        ['prequest']
            .concat(logPathname.split('/').filter((item) => item !== ''))
            .concat(logSuffix || [])
            .join('.')
    );

    const instance = got.extend({
        mutableDefaults: true,
        method,
        timeout,
        responseType,
        http2: false,
        headers: {
            'X-Request-Id': logID,
            'Content-type': 'application/json'
        },
        retry: {
            limit: 2,
            methods: ['GET', 'PUT', 'HEAD', 'DELETE', 'OPTIONS', 'TRACE', 'POST'],
            statusCodes: [408, 413, 500, 502, 503, 504, 521, 522, 524]
        },
        throwHttpErrors: false,
        hooks: {
            beforeRequest: [
                (options = {}) => {
                    const {headers = {}} = options;

                    if (!headers['x-request-id']) {
                        //TODO: removeme after PASSP-11180 is closed
                        logger.warn('No x-request-id found in request headers (PASSP-11180)');
                    }
                }
            ],
            afterResponse: [
                (response = {}) => {
                    logger.info(
                        'API: %s %s %j => [success] %j',
                        method.toUpperCase(),
                        logPathname,
                        logInput,
                        tools.clearLog(response.body)
                    );

                    return response;
                }
            ],
            beforeRetry: [
                (_options, error, retryCount) => {
                    const logUrl = `${method}:${logPathname}`;

                    logger.info(
                        'retry %d for %s with %j because of %s',
                        retryCount,
                        logUrl,
                        logInput,
                        error.toString()
                    );
                }
            ],
            beforeError: [
                (error) => {
                    const {response = {}} = error;
                    const {statusCode} = response;

                    if (![400, 404, 451, 429].includes(statusCode)) {
                        logger.error(
                            'API: %s %s %j => [reject] %s',
                            method.toUpperCase(),
                            logPathname,
                            logInput,
                            error.toString()
                        );
                    } else {
                        logger.info(
                            'API: %s %s %j => [reject] %s',
                            method.toUpperCase(),
                            logPathname,
                            logInput,
                            error.toString()
                        );
                    }

                    return error;
                }
            ]
        }
    });

    if (hooks) {
        instance.defaults.options = got.mergeOptions(instance.defaults.options, {hooks});
    }

    if (form) {
        requestParams.form = _.pickBy(form, (item) => typeof item !== 'undefined');
    }

    if (body) {
        if (typeof body === 'object' && !tools.isInstanceOfFormData(body)) {
            requestParams.body = JSON.stringify(body);
        } else {
            requestParams.body = body;
        }
    }

    return instance(url, requestParams);
};
