const inherit = require('inherit');
const assert = require('assert');
const Url = require('url');
const lodash = require('lodash');
const RecursiveRouter = require('./RecursiveRouter');
const trimSlashesRegexp = /^\/+|\/+$/g;

/**
 * @extends RecursiveRouter
 * @type {*}
 */
module.exports = inherit(RecursiveRouter, {
    __constructor: function() {
        this._regexps = {};
        this._map = {};
        this._default = undefined;
        this._strict = true;
    },

    setLax: function() {
        this._strict = false;
        return this;
    },

    _shortenLastSlash: function(url) {
        url = url.replace(trimSlashesRegexp, '').split('/');
        url.pop();
        return '/' + url.join('/');
    },

    _match: function(url) {
        url = Url.parse(url).pathname;
        url = '/' + url.replace(trimSlashesRegexp, '');

        var matched = this._map[url] || this._matchRe(url);

        if (typeof matched !== 'undefined') {
            return new this.__self.Match(matched, url);
        }

        if (url !== '/') {
            matched = this._match(this._shortenLastSlash(url));
            if (!this._strict || matched.getResult() instanceof RecursiveRouter) {
                return matched;
            }
        }

        if (typeof this._default !== 'undefined') {
            return new this.__self.Match(this._default, url);
        }

        throw new URIError('Unknown url given with no default defined');
    },

    _matchPage: function(url) {
        url = Url.parse(url).pathname;
        url = '/' + url.replace(trimSlashesRegexp, '');

        var matched = this._map[url] || this._matchRe(url);

        if (typeof matched !== 'undefined') {
            return matched;
        }

        if (url !== '/') {
            matched = this._matchPage(this._shortenLastSlash(url));
            if (!this._strict || matched instanceof RecursiveRouter) {
                return matched;
            }
        }

        if (typeof this._default !== 'undefined') {
            return this._default;
        }

        throw new Error('Unknown url given with no default defined');
    },

    _matchRe: function(url) {
        var matchingRe = lodash.find(this._regexps, (re) => (re.test(url) ? this._map[re.toString()] : false), this);

        return matchingRe ? this._map[matchingRe.toString()] : undefined;
    },

    map: function(url, value) {
        assert(typeof url === 'string' || url instanceof RegExp, 'Url should be a string or a regexp');
        assert(typeof value !== 'undefined', 'Mapped value should be defined. Perhaps use default?');
        assert(!(url in this._map), 'Value already mapped to the given key');

        if (url instanceof RegExp) {
            var regExp = url;

            url = regExp.toString();
            this._regexps[url] = regExp;
        }

        this._map[url] = value;

        return this;
    },

    default: function(value) {
        assert(typeof this._default === 'undefined', 'Value already defined as a default');
        assert(typeof value !== 'undefined', 'Value should be defined');

        this._default = value;

        return this;
    },

    transformUrl: function(originalUrl, matchedUrl) {
        originalUrl = '/' + Url.parse(originalUrl).pathname.replace(trimSlashesRegexp, '');
        return '/' + originalUrl.replace(matchedUrl, '').replace(trimSlashesRegexp, '');
    }
});
