var _ = require('lodash');
var assert = require('assert');

var skipPercentRegexp = /^%(?!\d)/;
var substitutionsRegexp = /%\d/g;

var knownKeysets = null;
var allowedLangs = null;
var loadedTranslations = null;

var isLangCode = function(lang) {
    return typeof lang === 'string' && lang.length === 2;
};

/**
 * Return a localized string
 * @param {string} lang         Language code to localize to
 * @param {string} key          Translation key
 * @param {string...} [args]    Replacement args
 */
var i18n = function(lang, key) {
    assert(isLangCode(lang), 'Lang should be a two-letter language code');
    assert(key && typeof key === 'string', 'Key should be a string');

    key = key.replace(skipPercentRegexp, '');

    var translations = i18n.getTranslations(lang);

    assert(key in translations, 'Unknown localization key: ' + key);

    var value = translations[key];
    var replacements = Array.prototype.slice.call(arguments, 2);

    return value.replace(substitutionsRegexp, function() {
        assert(replacements.length > 0, 'Substitution failed: not enough arguments');
        return replacements.shift();
    });
};

/**
 * Returns a dictionary with translations
 * @param {string} lang     Language code to translate to
 * @returns {Object}
 */
i18n.getTranslations = function(lang) {
    assert(loadedTranslations !== null, 'Localization files should be loaded with i18n.loadFromDir()');
    lang = allowedLangs[allowedLangs.indexOf(lang)] || 'ru';

    return loadedTranslations[lang];
};

i18n.setAllowedLangs = function(langs) {
    assert(
        _.isArray(langs) && langs.length > 0 && _.every(langs, isLangCode),
        'Langs should be an array of two-letter language codes'
    );

    allowedLangs = langs;
    return i18n;
};

i18n.setKeysets = function(keysets) {
    assert(
        _.isArray(keysets) &&
            keysets.length > 0 &&
            _.every(keysets, function(keyset) {
                return keyset && typeof keyset === 'string';
            }),
        'Keysets should be an array of keyset names matching the keyset part ' +
            'in the filename (Common for Common.ru.json)'
    );

    knownKeysets = _.uniq((knownKeysets || []).concat(keysets));
    return i18n;
};

i18n.loadFromDir = function(path) {
    assert(knownKeysets !== null, 'Before loading, please define keysets to look for using i18n.setKeysets([...])');
    assert(allowedLangs !== null, 'Before loading, please define allowed langs using i18n.setAllowedLangs([...])');

    var pathMessage = 'Provide a path to a directory with localization files';

    assert(path && path !== '/' && typeof path === 'string', pathMessage);

    path = path.replace(/\/$/, ''); // Removing trailing slash if there is one

    // eslint-disable-next-line no-unused-vars
    var files;

    try {
        files = require('fs').readdirSync(path);
    } catch (e) {
        throw new Error(pathMessage);
    }

    //Create {'ru': undefined, 'en': undefined}
    loadedTranslations = loadedTranslations || _.zipObject(allowedLangs);

    allowedLangs.forEach(function(lang) {
        loadedTranslations[lang] = _.merge.apply(
            null,
            knownKeysets
                .map(function(keyset) {
                    try {
                        return require(path + '/' + require('util').format('%s.%s.json', keyset, lang))[lang];
                    } catch (e) {
                        return {};
                    }
                })
                .concat([loadedTranslations[lang]])
        );
    });

    return i18n;
};

i18n.isConfigured = function() {
    return loadedTranslations !== null;
};

i18n.reset = function() {
    // For testing
    knownKeysets = null;
    allowedLangs = null;
    loadedTranslations = null;
};

module.exports = i18n;
