/* jshint forin:true, noempty:true, eqeqeq:true, bitwise: false, strict:true, browser:true */
/**
 * @module XorCipher
 *
 * This module provides simple encryption using XOR
 * its also encodes encrypted data using base64 encoding
 */
(function(global, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else {
        global.simpleCipher = factory();
    }
})(this, function() {
    'use strict';

    var b64Table = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';
    /**
     * @private
     * @param {Array<number>} - array of unicode codes
     * @returns {String} base64 string
     */

    function b64Encode(data) {
        var o1;
        var o2;
        var o3;
        var h1;
        var h2;
        var h3;
        var h4;
        var bits;
        var r;
        var i = 0;
        var enc = '';

        if (!data) {
            return data;
        }
        do {
            o1 = data[i++];
            o2 = data[i++];
            o3 = data[i++];
            bits = (o1 << 16) | (o2 << 8) | o3;
            h1 = (bits >> 18) & 0x3f;
            h2 = (bits >> 12) & 0x3f;
            h3 = (bits >> 6) & 0x3f;
            h4 = bits & 0x3f;
            enc += b64Table.charAt(h1) + b64Table.charAt(h2) + b64Table.charAt(h3) + b64Table.charAt(h4);
        } while (i < data.length);
        r = data.length % 3;
        return (r ? enc.slice(0, r - 3) : enc) + '==='.slice(r || 3);
    }
    /**
     * @private
     * @param {String} base64 string
     * @returns {Array<number>} - array of unicode codes
     */
    function b64Decode(data) {
        var o1;
        var o2;
        var o3;
        var h1;
        var h2;
        var h3;
        var h4;
        var bits;
        var i = 0;
        var result = [];

        if (!data) {
            return data;
        }
        data += '';
        do {
            h1 = b64Table.indexOf(data.charAt(i++));
            h2 = b64Table.indexOf(data.charAt(i++));
            h3 = b64Table.indexOf(data.charAt(i++));
            h4 = b64Table.indexOf(data.charAt(i++));
            bits = (h1 << 18) | (h2 << 12) | (h3 << 6) | h4;
            o1 = (bits >> 16) & 0xff;
            o2 = (bits >> 8) & 0xff;
            o3 = bits & 0xff;
            result.push(o1);
            if (h3 !== 64) {
                result.push(o2);
                if (h4 !== 64) {
                    result.push(o3);
                }
            }
        } while (i < data.length);
        return result;
    }
    /**
     * @private
     * @param {String} secret
     * @param {Number} index of symbol in secret
     * @returns {Number} unicode code
     */
    function keyCharAt(secret, i) {
        return secret.charCodeAt(Math.floor(i % secret.length));
    }
    /**
     * @private
     * @param {String} secret
     * @param {String} data
     * @returns {Array} array with unicode codes
     */
    function xorEncrypt(secret, data) {
        return data.split('').map(function(c, i) {
            return c.charCodeAt(0) ^ keyCharAt(secret, i);
        });
    }
    /**
     * @private
     * @param {String} secret
     * @param {String} data
     * @returns {String}
     */
    function xorDecrypt(secret, data) {
        return data
            .map(function(c, i) {
                return String.fromCharCode(c ^ keyCharAt(secret, i));
            })
            .join('');
    }
    /**
     * Check whether string contains only ascii symbols
     *
     * @param {String}
     * @returns {boolean}
     *
     */
    function isAscii(str) {
        if (typeof str !== 'string') {
            return false;
        }
        for (var i = 0; i < str.length; i++) {
            if (str.charCodeAt(i) > 127) {
                return false;
            }
        }
        return true;
    }

    return {
        /**
         * Encrypt text with xor cipher and base64 encode it
         *
         * @param {String} secret - cipher key
         * @param {String} data - data to encrypt
         *
         * @returns {String}
         */
        encode: function(secret, data) {
            if (isAscii(secret) && isAscii(data)) {
                data = xorEncrypt(secret, data);
                return b64Encode(data);
            } else {
                throw new Error('Secret and data should be ASCII string');
            }
        },
        /**
         * Decrypt text with xor cipher and base64 decode it
         *
         * @param {String} secret - cipher key
         * @param {String} data - data to decrypt
         *
         * @returns {String}
         */
        decode: function(secret, data) {
            if (isAscii(secret) && isAscii(data)) {
                data = b64Decode(data);
                return xorDecrypt(secret, data);
            } else {
                throw new Error('Secret and data should be ASCII string');
            }
        }
    };
});
