var _ = require('lodash');
var when = require('when');
var assert = require('assert');
var PLog = require('plog');
const yr = require('yate/lib/runtime');

/**
 * Renderer method
 * Receives the templates dir path, template filename and options object
 *
 * @param {string}  templatesDirPath    Absolute path to the templates directory
 * @param {string}  filename            Template filename in the templates dir
 * @param {object}  data                Data to render the templates with
 * @type {Function}
 */
var renderer = null;

/**
 * Base directory to search templates in
 * @type {string}
 */
var templatesDir = null;

var _checkDir = function(dir) {
    assert(
        dir && typeof dir === 'string' && dir !== '/',
        'Templates dir is expected to be a path to templates directory'
    );
};

/**
 * Stands for things that get rendered
 * Transforms the data for a format suitable for yate
 *
 * @typedef View
 * @class View
 * @constructor
 */
module.exports = require('inherit')(
    {
        name: null, //Should be defined by views

        __constructor: function() {
            assert(this.name && typeof this.name === 'string', 'Please define a name for your view for logging');
            this._logger = new PLog(null, 'oauth', 'view', this.name);
        },

        _compile: function() {
            throw new Error('Should be overwritten to return templating data of your choice');
        },

        /**
         * Compile the view into a data object to pass to the templating engine
         */
        compile: function() {
            this._logger.debug('Compiling');
            this._logger.verbose('Compiling with %j', arguments);
            return when.resolve(this._compile.apply(this, arguments));
        },

        /**
         * Template file name to render, for example: 'index.ru.js'
         * @returns {string}
         */
        getTemplate: function() {
            throw new Error('Should be overwritten to return the template filename, example: index.ru.js');
        },

        /**
         * Transform the view into the HTML
         * @returns {string}
         */
        render: function() {
            assert(renderer !== null, 'Renderer should be defined with View.setRenderer method');
            assert(templatesDir !== null, 'Templates dir path should be defined with View.setTemplatesDir');

            var that = this;
            var args = arguments;

            return this.compile.apply(this, args).then(function(compiled) {
                var template = that.getTemplate.apply(that, args);

                that._logger.debug('Rendering %s', template);
                that._logger.verbose('Rendering the %s with %j', template, compiled);
                return renderer(templatesDir, template, compiled);
            });
        }
    },
    {
        /**
         * Set renderer method
         * @param {Function} method
         */
        setRenderer: function(method) {
            assert(
                typeof method === 'function',
                'Renderer should be a function accepting templatesDir path, filename and templating data'
            );
            renderer = method;
            return this;
        },

        setTemplatesDir: function(templatesDirPath) {
            _checkDir(templatesDirPath);
            templatesDir = templatesDirPath.replace(/\/$/, '');
            return this;
        },

        yateRenderer: (function() {
            var extRe = /\.js$/;

            return function(templatesDirPath, filename, data) {
                _checkDir(templatesDirPath);
                assert(
                    filename && typeof filename === 'string' && extRe.test(filename),
                    'Filename should be a name of the template file'
                );
                assert(_.isObjectLike(data), 'Templating data should be a plain object');

                var templatePath = templatesDirPath + '/' + filename;

                PLog.verbose()
                    .type('oauth', 'renderer', 'yate')
                    .write('Requiring %s', templatePath);
                require(templatePath);

                var mode = filename.replace(extRe, '');

                PLog.verbose()
                    .type('oauth', 'renderer', 'yate')
                    .write('Rendering mode %s with %j', mode, data);

                try {
                    return yr.run(mode, data);
                } catch (e) {
                    if (e instanceof Error) {
                        throw e;
                    }

                    //Transforming arbitrary stuff throw into an Error
                    throw new Error(e);
                }
            };
        })(),

        getYateRenderer: function() {
            return yr;
        },

        reset: function() {
            //For testing
            templatesDir = null;
            renderer = null;
        }
    }
);
