/* eslint-disable no-console */
const request = require('request');
const when = require('when');
const fs = require('fs');
const path = require('path');
const baseRequest = request.defaults({
    headers: {Authorization: `OAuth ${process.env.TANKER_TOKEN}`}
});

module.exports = class TankerSync {
    constructor(options) {
        this._setOptions(options);
        this._setHistory();
        this._setConfig();

        this._keySetsPromises = [];
        this._alreadySentKeySetsPromises = {};
    }

    run() {
        const {config, _keySetsPromises, _alreadySentKeySetsPromises} = this;

        for (let bundleName in config) {
            if (!config.hasOwnProperty(bundleName)) {
                return;
            }

            const bundleConfig = config[bundleName];
            const {keySets, projectName, branchName, forceLoad} = bundleConfig;

            bundleConfig.keySetsPromises = [];

            keySets.forEach((keySet) => {
                const historyPromiseName = `${projectName}.${branchName}.${keySet}`;

                let loadHistoryPromise;

                if (_alreadySentKeySetsPromises[historyPromiseName] && !forceLoad) {
                    loadHistoryPromise = _alreadySentKeySetsPromises[historyPromiseName];
                    bundleConfig.keySetsPromises.push(loadHistoryPromise.promise);
                    return;
                }

                loadHistoryPromise = when.defer();
                _alreadySentKeySetsPromises[historyPromiseName] = loadHistoryPromise;
                _keySetsPromises.push(loadHistoryPromise.promise);
                bundleConfig.keySetsPromises.push(loadHistoryPromise.promise);

                if (forceLoad) {
                    const historyHashPath = `${bundleName}.${projectName}.${branchName}.${keySet}`;

                    delete this.history[historyHashPath];
                    loadHistoryPromise.resolve(true);
                    return;
                }

                this._loadKeySetHistory(bundleName, bundleConfig, keySet).then((keySetWasChanged) =>
                    loadHistoryPromise.resolve(keySetWasChanged)
                );
            });

            when.all(bundleConfig.keySetsPromises).then((keySetsState) => {
                const historyWasChanged = keySetsState.some((keySetWasChanged) => keySetWasChanged);

                if (!historyWasChanged) {
                    return;
                }

                this._loadKeySetLocs(bundleConfig, keySets.join(','));
            });
        }

        when.all(_keySetsPromises).then(() => {
            console.log('\x1b[32m%s\x1b[0m', '[HISTORY SAVED]');

            fs.writeFile(path.resolve(this.options.historyPath), JSON.stringify(this.history), 'utf8', () => {});
        });
    }

    _loadKeySetHistory(bundleName, bundleConfig, keySet) {
        const {projectName, branchName} = bundleConfig;
        const historyUrlParams = [
            `project-id=${projectName}`,
            `branch-id=${branchName}`,
            `keyset-id=${keySet}`,
            'limit=1'
        ];
        const url = `https://${this.options.host}/fhistory/?${historyUrlParams.join('&')}`;

        console.log('\x1b[33m%s\x1b[0m', `[CHECK HISTORY CHANGES]`, url);

        const promise = when.defer();

        baseRequest.get(url, (error, res, body) => {
            let data;

            try {
                data = JSON.parse(body);
            } catch (error) {
                console.log('\x1b[31m%s\x1b[0m', '[PARSE HISTORY FROM SERVER ERROR]');
                console.log(body);
                console.error(error);
                return promise.resolve(true);
            }

            const commits = data.commits;

            if (!(commits && commits.length)) {
                return promise.resolve(true);
            }

            const hash = commits[0].sha1;
            const historyHashPath = `${bundleName}.${projectName}.${branchName}.${keySet}`;
            const historyHash = this.history[historyHashPath];

            if (!historyHash || historyHash !== hash) {
                this.history[historyHashPath] = hash;
                return promise.resolve(hash);
            }

            return promise.resolve();
        });

        return promise.promise;
    }

    _loadKeySetLocs(bundleConfig, keySet) {
        const languages = bundleConfig.languages ? bundleConfig.languages : this.options.languages;

        bundleConfig.formats.forEach((format) => {
            if (bundleConfig.needMerge && format !== 'json') {
                console.log(
                    '\x1b[31m%s\x1b[0m',
                    '[PARSE HISTORY FROM SERVER ERROR]',
                    `Keyset '${keySet}' can't be merged in format '${format}'.`
                );

                return;
            }

            if (format === 'json') {
                const url = this._getKeySetLoadUrl(bundleConfig, keySet, format, languages);

                baseRequest.get(url, (error, res, body) => {
                    this._writeLocsToJson(bundleConfig, body);
                });
            } else if (format === 'loc') {
                languages.forEach((lang) => {
                    const url = this._getKeySetLoadUrl(bundleConfig, keySet, format, [lang]);

                    baseRequest.get(url, (error, res, body) => {
                        this._writeLocsToLocExt(bundleConfig, body, lang);
                    });
                });
            } else if (format === 'js') {
                languages.forEach((lang) => {
                    const url = this._getKeySetLoadUrl(bundleConfig, keySet, format, [lang]);

                    baseRequest.get(url, (error, res, body) => {
                        this._writeLocsToJSExt(bundleConfig, body, lang);
                    });
                });
            }
        });
    }

    _getKeySetLoadUrl(bundleConfig, keySet, format, langs) {
        const {projectName, branchName, options, loadUrl, needMerge} = bundleConfig;
        const params = [
            `project-id=${projectName}`,
            `branch-id=${branchName}`,
            `keyset-id=${keySet}`,
            `language=${langs.join(',')}`,
            `safe`
        ];

        for (let optionKey in options) {
            if (!options.hasOwnProperty(optionKey)) {
                return;
            }

            params.push(`${optionKey}=${options[optionKey]}`);
        }

        let url;

        if (loadUrl) {
            url = `${loadUrl}?${params.join('&')}`;
        } else {
            const method = needMerge ? 'projects/export' : 'keysets';

            url = `https://${this.options.host}/${method}/${format}/?${params.join('&')}`;
        }

        console.log('\x1b[35m%s\x1b[0m', '[LOAD LOCALIZATION]', url);
        return url;
    }

    _writeLocsToJson(bundleConfig, data) {
        const {exportFile, afterSaveCallback, needMerge} = bundleConfig;
        const languages = bundleConfig.languages ? bundleConfig.languages : this.options.languages;

        let locs;

        try {
            locs = JSON.parse(data);
        } catch (error) {
            console.log(data);
            console.log('\x1b[31m%s\x1b[0m', '[PARSE LOCALIZATION FROM SERVER ERROR]');
            console.error(error);
            process.exit(1);
        }

        if (needMerge) {
            const outputFileName = path.resolve(this.options.outputPath, exportFile.replace('${FORMAT}', 'json'));

            console.log('\x1b[36m%s\x1b[0m', '[WRITE .JSON]', outputFileName);

            fs.writeFile(outputFileName, JSON.stringify(locs), 'utf8', () => {
                if (afterSaveCallback && typeof afterSaveCallback === 'function') {
                    afterSaveCallback(outputFileName);
                }
            });

            return;
        }

        languages.forEach((lang) => {
            const langLocs = {};
            const outputFileName = path.resolve(
                this.options.outputPath,
                exportFile.replace('${LNG}', lang).replace('${FORMAT}', 'json')
            );

            console.log('\x1b[36m%s\x1b[0m', '[WRITE .JSON]', outputFileName);
            langLocs[lang] = locs[lang];

            fs.writeFile(outputFileName, JSON.stringify(langLocs), 'utf8', () => {
                if (afterSaveCallback && typeof afterSaveCallback === 'function') {
                    afterSaveCallback(outputFileName);
                }
            });
        });
    }

    _writeLocsToLocExt(bundleConfig, data, lang) {
        const {exportFile, afterSaveCallback} = bundleConfig;
        const outputFileName = path.resolve(
            this.options.outputPath,
            exportFile.replace('${LNG}', lang).replace('${FORMAT}', 'loc')
        );

        console.log('\x1b[36m%s\x1b[0m', '[WRITE .LOC]', outputFileName);

        fs.writeFile(outputFileName, data, 'utf8', () => {
            if (afterSaveCallback && typeof afterSaveCallback === 'function') {
                afterSaveCallback(outputFileName);
            }
        });
    }

    _writeLocsToJSExt(bundleConfig, data, lang) {
        const {exportFile, afterSaveCallback} = bundleConfig;
        const outputFileName = path.resolve(
            this.options.outputPath,
            exportFile.replace('${LNG}', lang).replace('${FORMAT}', 'js')
        );

        console.log('\x1b[36m%s\x1b[0m', '[WRITE .JS]', outputFileName);

        fs.writeFile(outputFileName, data, 'utf8', () => {
            if (afterSaveCallback && typeof afterSaveCallback === 'function') {
                afterSaveCallback(outputFileName);
            }
        });
    }

    _setOptions(options) {
        this._validateOptions(options);

        const defaultOptions = {
            host: 'tanker-api.yandex-team.ru',
            languages: ['ru', 'en', 'tr', 'uk', 'id', 'fr', 'fi']
        };

        if (!options.host) {
            options.host = defaultOptions.host;
        }

        if (!options.languages || !options.languages.length) {
            options.languages = defaultOptions.languages;
        }

        this.options = options;
    }

    _validateOptions(options) {
        const availableOptions = ['host', 'languages', 'configPath', 'historyPath', 'outputPath'];
        const requiredOptions = ['configPath', 'historyPath', 'outputPath'];

        try {
            if (requiredOptions.some((optionName) => !options[optionName])) {
                throw new Error(`TankerSync: all of ${requiredOptions} is required.`);
            }

            for (let optionName in options) {
                if (!options.hasOwnProperty(optionName)) {
                    return;
                }

                if (availableOptions.indexOf(optionName) === -1) {
                    throw new Error(`TankerSync: Option ${optionName} is not available.`);
                }
            }
        } catch (error) {
            console.log(error);
            process.exit(1);
        }
    }

    _setHistory() {
        try {
            const historyObject = require(path.resolve(this.options.historyPath));

            let history = JSON.parse(JSON.stringify(historyObject));

            if (typeof history !== 'object') {
                history = {};
            }

            this.history = history;
        } catch (error) {
            console.log('\x1b[31m%s\x1b[0m', '[PARSE HISTORY ERROR]');
            console.error(error);
            process.exit(1);
        }
    }

    _validateConfig(config) {
        if (typeof config !== 'object') {
            throw new Error(`TankerSync: Config should be an object.`);
        }

        for (let bundleName in config) {
            if (!config.hasOwnProperty(bundleName)) {
                continue;
            }

            const bundleConfig = config[bundleName];
            const {
                exportFile,
                projectName,
                keySets,
                formats,
                options,
                afterSaveCallback,
                loadUrl,
                branchName,
                languages
            } = bundleConfig;

            if (!exportFile || typeof exportFile !== 'string') {
                this._throwInvalidConfigParamError('exportFile', bundleName);
            }

            if (!projectName || typeof projectName !== 'string') {
                this._throwInvalidConfigParamError('projectName', bundleName);
            }

            if (!keySets || !Array.isArray(keySets)) {
                this._throwInvalidConfigParamError('keySets', bundleName);
            }

            if (!formats || !Array.isArray(formats)) {
                this._throwInvalidConfigParamError('formats', bundleName);
            }

            if (!options || typeof options !== 'object') {
                this._throwInvalidConfigParamError('options', bundleName);
            }

            if (afterSaveCallback && typeof afterSaveCallback !== 'function') {
                this._throwInvalidConfigParamError('afterSaveCallback', bundleName);
            }

            if (loadUrl && typeof loadUrl !== 'string') {
                this._throwInvalidConfigParamError('loadUrl', bundleName);
            }

            if (languages && !Array.isArray(languages)) {
                this._throwInvalidConfigParamError('languages', bundleName);
            }

            if (!branchName) {
                bundleConfig.branchName = 'master';
            }
        }
    }

    _throwInvalidConfigParamError(paramName, bundleName) {
        let errorMessage = `TankerSync: Invalid param '${paramName}' for bundle '${bundleName}'.`;

        throw new Error(errorMessage);
    }

    _setConfig() {
        try {
            const config = require(path.resolve(this.options.configPath));

            this._validateConfig(config);
            this.config = config;
        } catch (error) {
            console.error(error);
            process.exit(1);
        }
    }
};
