/* eslint-disable no-console */
const cp = require('child_process');

const kill = require('tree-kill');

const hideTokens = data => data.replace(/AQAD-[\w]{39}/g, `AQAD-${'*'.repeat(36)}`);

/**
 * Компонент, запускающий некую команду в шэлле.
 * @param {String} name Имя таска для Archon.
 * @param {[String]} deps Список зависимостей для Archon graph.
 * @param {Object} options Опции, в основном для child_process
 * [https://nodejs.org/docs/latest-v8.x/api/child_process.html].
 * @param {Boolean} options.async Флаг должен ли процесс считаться завершенным сразу синхронно, или он продолжит работать в фоне (true).
 * @param {Number} options.asyncWait Сколько времени нужно подождать асинхронного процесса, прежде чем заявлять, что процесс "завершен".
 * @param {String} options.cmd Команда, которую нужно запустить.
 * @param {Number} options.initTimeout  Таймаут выполнения init. Если таймаут равен 0, то операция не лимитируется по времени.
 * @param {[String]} options.args Список аргументов для команды.
 * @param {[String]} options.env Список переменных окружения для команды.
 * @param {String} options.cwd Директория, в которой будет исполнена команда.
 * @param {Boolean|String} options.shell В какой оболочке запускать команду. Подробнее https://nodejs.org/api/child_process.html#child_process_child_process_spawn_command_args_options.
 * @param {String|Array} options.stdio Child's stdio configuration.
 * @param {String} options.stdin Child's stdin content. Conflicts with stdio
 */
class Cmd {
  constructor({ name = 'cmd', deps = [], options = {} }) {
    this.name = name;
    this.deps = deps;
    this.initTimeout = options.initTimeout || 120000;
    this.options = options;

    this._proc = null;
    this._logName = hideTokens(`${options.cmd} ${options.args && options.args.join(' ')}`);
  }

  async init() {
    const { cmd, args, env, cwd = process.cwd(), stdio, ignoreCodes } = this.options;
    let { stdin } = this.options;

    let stdinConfig = stdio;

    if (!stdinConfig) {
      if (!stdin) {
        stdinConfig = 'inherit';
      } else {
        stdinConfig = ['pipe', 1, 2];
      }
    }

    return new Promise((resolve, reject) => {
      console.log(`> [${this._logName}] Spawning...`);

      this._proc = cp.spawn(cmd, args, {
        env: { ...process.env, ...env },
        stdio: stdinConfig,
        cwd,
        shell: this.options.shell || false,
      });

      if (stdin) {
        if (typeof stdin === 'function') {
          stdin = stdin();
        }
        this._proc.stdin.end(stdin);
      }

      this._proc.on('error', e => {
        console.error(e);

        reject();
      });

      this._proc.on('exit', (code, signal) => {
        clearTimeout(this._asyncReporter);
        if (code && (!ignoreCodes || !ignoreCodes[code])) {
          if (code == null) {
            if (signal !== 'SIGTERM' || !this._gracefullStop) {
              const error = `${this.name} exited because of ${signal}`;

              console.log(`! [${this._logName}] Error! ${error}`);

              return reject(error);
            }
          } else if (code !== 0) {
            const error = `${this.name} exited with code ${code}`;

            console.log(`! [${this._logName}] Error! ${error}`);

            return reject(error);
          }
        }

        console.log(`< [${this._logName}] Done!`, code !== 0 ? `(with ignored code ${code})` : '');
        resolve();
      });

      if (!this.options.async) {
        return;
      }

      const asyncWait = this.options.asyncWait || 0;

      this._asyncReporter = setTimeout(() => {
        console.log(`< [${this._logName}] Resolved after ${this.options.asyncWait} [probably still running]!`);
        resolve();
      }, asyncWait);
    });
  }

  async _kill(signal) {
    const { killType = 'SIGINT' } = this.options;

    if (!this._proc) {
      return;
    }

    return new Promise((resolve, reject) => {
      kill(
        this._proc.pid,
        signal || killType,
        err => {
          if (err) {
            return reject(err);
          }

          resolve();
        },
      );
    });
  }

  async stop() {
    if (!this._proc) {
      return;
    }

    this._gracefullStop = true;
    clearTimeout(this._asyncReporter);
    await this._kill();
  }

  async shutdown() {
    if (!this._proc) {
      return;
    }

    clearTimeout(this._asyncReporter);
    await this._kill();
  }

  async terminate() {
    if (!this._proc) {
      return;
    }

    clearTimeout(this._asyncReporter);
    await this._kill();
  }
}

module.exports.Cmd = Cmd;
