import fs from 'fs';
import path from 'path';
import { promisify } from 'util';

import { paramCase } from 'param-case';

import type { Style } from '@figma-export/types';

import { exportStyles } from './figma';

const writeFile = promisify(fs.writeFile);
const mkdir = promisify(fs.mkdir);

const FIGMA_TOKEN = process.env.FIGMA_TOKEN;

if (!FIGMA_TOKEN) {
  throw new Error('FIGMA_TOKEN is required!');
}

const FIGMA_FILES = {
  COLOR_LIGHT: 'zTfXeAaSskHEQo471WBCK0',
  COLOR_DARK: '7GELhKJbYRZv6hrp2Hulez',
  TYPOGRAPHY: 'IJcCPtaIAuUC31CyO8cs88',
};

const THEME_DIR = path.resolve(__dirname, '../src/themes');

function getVariableName(style: Style) {
  const name = paramCase(style.name);

  switch (style.styleType) {
    case 'FILL':
      return `id-color-${name}`;

    case 'TEXT':
      return `id-typography-${name}`;

    case 'EFFECT':
      return `id-${name}`;
  }

  return name;
}

async function exportColorStyle(file: string, themeName: string, colorSchema: 'light' | 'dark') {
  return exportStyles({
    token: FIGMA_TOKEN,
    file,
    formatters: [
      {
        name: 'css/variable',
        options: {
          selector: `[data-theme='${themeName}'][data-color='${colorSchema}']`,
          destination: path.join(THEME_DIR, themeName, `tokens/color-${colorSchema}.css`),
          filter: (style: Style) => style.styleType === 'FILL',
          getVariableName,
        },
      },
      {
        name: 'css/variable',
        options: {
          selector: `[data-theme='${themeName}'][data-color='${colorSchema}']`,
          destination: path.join(THEME_DIR, themeName, `tokens/effect-${colorSchema}.css`),
          filter: (style: Style) => style.styleType === 'EFFECT',
          getVariableName,
        },
      },
    ],
  });
}

async function exportTypography(file: string, themeName: string) {
  return exportStyles({
    token: FIGMA_TOKEN,
    file,
    formatters: [
      {
        name: 'css/variable',
        options: {
          selector: `[data-theme='${themeName}']`,
          destination: path.join(THEME_DIR, themeName, 'tokens/typography.css'),
          filter: (style) => {
            if (style.styleType !== 'TEXT') {
              return false;
            }

            const name = paramCase(style.name);

            return ['regular', 'medium', 'bold'].every((weight) => !name.includes(weight));
          },
          getVariableName,
        },
      },
    ],
  });
}

async function createThemeFile(name: string) {
  const content = [
    `/**\n * Do not edit directly.\n **/`,
    "import './tokens/typography.css';",
    "import './tokens/color-light.css';",
    "import './tokens/color-dark.css';",
    "import './tokens/effect-light.css';",
    "import './tokens/effect-dark.css';",
    '',
  ];

  const filePath = path.join(THEME_DIR, name, 'index.ts');

  await mkdir(path.dirname(filePath), { recursive: true });
  await writeFile(filePath, content.join('\n'));
}

async function createTheme(name: string) {
  const themeName = paramCase(name);

  const promises: Array<Promise<void>> = [
    exportColorStyle(FIGMA_FILES.COLOR_LIGHT, themeName, 'light'),
    exportColorStyle(FIGMA_FILES.COLOR_DARK, themeName, 'dark'),
    exportTypography(FIGMA_FILES.TYPOGRAPHY, themeName),
    createThemeFile(themeName),
  ];

  await Promise.all(promises);

  // eslint-disable-next-line no-console
  console.log(`The theme ${themeName} has been successfully created`);
}

createTheme('default');
