import fs from 'fs';
import path from 'path';
import { promisify } from 'util';

import { paramCase } from 'param-case';

import { Style, StyleTypeEffect, StyleTypeFill, StyleTypeText } from '@figma-export/types';

import { StylesFormatter } from '../types';

const writeFile = promisify(fs.writeFile);
const mkdir = promisify(fs.mkdir);

interface CssVariable {
  name: string;
  value: string | number;
  comment?: string;
}

export interface CssVariableFormatterOptions {
  selector?: string;
  destination: string;
  getVariableName?: (style: Style) => string;
  filter?: (style: Style) => boolean;
}

export class CssVariableFormatter implements StylesFormatter {
  private options: CssVariableFormatterOptions;

  constructor(options: CssVariableFormatterOptions) {
    this.options = options;
  }

  async format(styles: Style[]) {
    const selector = this.options.selector ?? ':root';
    const variables: CssVariable[] = [];

    for (const style of styles) {
      if (this.skip(style)) {
        continue;
      }

      for (const variable of this.getVariables(style)) {
        variables.push(variable);
      }
    }

    variables.sort((a, b) => a.name.localeCompare(b.name));

    const header = `/**
 * Do not edit directly.
 **/
`;

    const lines: string[] = [];

    for (const variable of variables) {
      if (variable.comment) {
        lines.push(`  /* ${variable.comment} */`);
      }

      lines.push(`  --${variable.name}: ${variable.value};`);
    }

    const content = `${header}\n${selector} {\n${lines.join('\n')}\n}\n`;

    await this.save(content);
  }

  private skip(style: Style) {
    const { filter = () => true } = this.options;

    return !style.visible || !filter(style);
  }

  private *getVariables(style: Style): Generator<CssVariable> {
    switch (style.styleType) {
      case 'FILL': {
        yield this.getStyleFillVariable(style);
        break;
      }

      case 'EFFECT': {
        yield this.getStyleEffectVariable(style);
        break;
      }

      case 'TEXT': {
        yield* this.getStyleTextVariables(style);
      }
    }
  }

  private getStyleFillVariable(style: Style & StyleTypeFill) {
    const values = style.fills.reduce<string[]>((acc, fill) => {
      if (fill.visible) {
        acc.push(fill.value);
      }

      return acc;
    }, []);

    return {
      name: this.getVariableName(style),
      value: values.join(', '),
      comment: style.comment,
    };
  }

  private getStyleEffectVariable(style: Style & StyleTypeEffect) {
    const boxShadowValues: string[] = [];
    const filterBlurValues: string[] = [];

    for (const effect of style.effects) {
      if (!effect.visible) {
        continue;
      }

      if (effect.type === 'INNER_SHADOW' || effect.type === 'DROP_SHADOW') {
        boxShadowValues.push(effect.value);
      }

      if (effect.type === 'LAYER_BLUR') {
        filterBlurValues.push(effect.value);
      }
    }

    const boxShadowValue = boxShadowValues.join(', ');
    const filterBlurValue = filterBlurValues.join(', ');
    const value = boxShadowValue || filterBlurValue;

    return {
      name: this.getVariableName(style),
      value,
      comment: style.comment,
    };
  }

  private *getStyleTextVariables(style: Style & StyleTypeText): Generator<CssVariable> {
    const name = this.getVariableName(style);
    const { fontFamily, fontSize, fontStyle, fontWeight, lineHeight } = style.style;

    const fonts: string[] = [`'${fontFamily}'`];

    if (fontFamily !== 'YS Text') {
      fonts.push(`'YS Text'`);
    }
    fonts.push('Arial', 'sans-serif');

    yield {
      name,
      value: `${fontStyle} ${fontWeight} ${fontSize}px/${lineHeight}px ${fonts.join(', ')}`,
      comment: style.comment,
    };
  }

  private getVariableName(style: Style) {
    const { getVariableName } = this.options;

    if (typeof getVariableName === 'function') {
      return getVariableName(style);
    }

    return paramCase(style.name);
  }

  private async save(content: string) {
    const { destination } = this.options;

    const filePath = path.resolve(destination);

    await mkdir(path.dirname(destination), { recursive: true });
    await writeFile(filePath, content);

    // eslint-disable-next-line no-console
    console.log(`Styles have been successfully saved: ${filePath}`);
  }
}
