'use strict';

const { I18N_IMPORT_REGEX } = require('../constants');

const isNodeInFunction = (node) => {
  while (node.parent) {
    // eslint-disable-next-line no-param-reassign
    node = node.parent;

    if (
      node.type === 'FunctionDeclaration' ||
      node.type === 'FunctionExpression' ||
      node.type === 'ArrowFunctionExpression'
    )
      return true;
  }
};

/**
 * @type {import('eslint').Rule.RuleModule}
 */
module.exports = {
  meta: {
    type: `problem`,
    docs: {
      description: 'Warns for top level i18n calls',
    },
    messages: {
      topLevelCall: 'Top level `i18n` call is forbidden, wrap it in function call',
    },
  },

  create: (context) => {
    const sourceCode = context.getSourceCode();
    const importedI18NNames = sourceCode.ast.body.reduce((acc, node) => {
      if (node.type !== 'ImportDeclaration') return acc;

      // `node.source.value` is a string like `import { i18n } from './${dirname}.i18n'`
      if (!I18N_IMPORT_REGEX.test(node.source.value)) return acc;

      node.specifiers.forEach((specifier) => {
        if (specifier.imported) {
          acc.push(specifier.imported.name);
        }
      });

      return acc;
    }, []);

    return {
      CallExpression: (node) => {
        if (!importedI18NNames.includes(node.callee.name) || isNodeInFunction(node)) return;

        context.report({
          node,
          messageId: 'topLevelCall',
        });
      },
    };
  },
};
