import { promises as fs } from 'fs';
import * as path from 'path';
import { env, cwd } from 'process';
import extract from '@yandex/figma-icon-extractor';
import { ComponentTemplateFn } from '@yandex/figma-icon-extractor/lib/svg-to-jsx';

const FIGMA_TOKEN = env.FIGMA_TOKEN;
const FIGMA_FILE = '4oNoOI8oxWPTEqDz84jaBa';
const BANKS_ICONS_PAGE = '1:25';
const SYSTEMS_ICONS_PAGE = '1:3512';

const ICONS_FOLDER = path.resolve(cwd(), 'src/icons');
const TMP_ICONS_FOLDER = path.resolve(cwd(), 'src/icons.tmp');

const BANK_SIZES = [24, 32];

const componentBankTemplateFn: ComponentTemplateFn = props => {
  const { svg, sizes, name } = props;

  let propsType;
  let propsDecl;

  if (sizes[0] === 0) {
    propsType = `IconProps<${BANK_SIZES.join(' | ')}>`;
    propsDecl = `const { className, size = ${BANK_SIZES[0]}, ...otherProps } = props;`;
  } else if (sizes.length === 1) {
    propsType = `Omit<IconProps<${sizes[0]}>, 'size'>`;
    propsDecl = `
    const { className, ...otherProps } = props;
    const size = ${sizes[0]};
    `;
  } else {
    propsType = `IconProps<${sizes.join(' | ')}>`;
    propsDecl = `const { className, size = ${sizes[0]}, ...otherProps } = props;`;
  }

  const template = `
    /* This file was created automatically, don't change it manually. */
    import React, { forwardRef } from 'react'
    import { IconProps } from '../types';
    export const ${name}Icon = forwardRef<SVGSVGElement, ${propsType}>((props, ref) => {
      ${propsDecl}
      return (
        ${svg}
      )
    })
  `;

  return template;
};

const componentSystemTemplateFn: ComponentTemplateFn = props => {
  const { svg, name } = props;

  const propsType = `IconProps`;
  const propsDecl = `const { className, ...otherProps } = props;`;

  const template = `
    /* This file was created automatically, don't change it manually. */
    import React, { forwardRef } from 'react'
    import { IconProps } from '../types';
    export const ${name}Icon = forwardRef<SVGSVGElement, ${propsType}>((props, ref) => {
      ${propsDecl}
      return (
        ${svg}
      )
    })
  `;

  return template;
};

const syncIcons = async() => {
  // Extract bank icons to temp directory
  await extract(TMP_ICONS_FOLDER, {
    token: FIGMA_TOKEN,
    file: FIGMA_FILE,
    page: BANKS_ICONS_PAGE,
    preserveColors: true,
    filter: 'tsx',
    componentTemplateFn: componentBankTemplateFn,
  });

  // Move icons from temp directory to target one
  await Promise.all(
    (await fs.readdir(TMP_ICONS_FOLDER)).map(iconPath =>
      fs.rename(`${TMP_ICONS_FOLDER}/${iconPath}`, `${ICONS_FOLDER}/${iconPath}`),
    ),
  );

  // Extract system icons to temp directory
  await extract(TMP_ICONS_FOLDER, {
    token: FIGMA_TOKEN,
    file: FIGMA_FILE,
    page: SYSTEMS_ICONS_PAGE,
    nonSquare: true,
    preserveColors: true,
    filter: 'tsx',
    componentTemplateFn: componentSystemTemplateFn,
    // exportTemplateFn,
  });

  // Move icons from temp directory to target one
  await Promise.all(
    (await fs.readdir(TMP_ICONS_FOLDER))
      .filter(path => !path.endsWith('ts')) // filter index file
      .map(iconPath => fs.rename(`${TMP_ICONS_FOLDER}/${iconPath}`, `${ICONS_FOLDER}/${iconPath}`)),
  );

  // Merge index files
  await fs.appendFile(`${ICONS_FOLDER}/index.ts`, await fs.readFile(`${TMP_ICONS_FOLDER}/index.ts`));

  // Remove temp directory
  await fs.rmdir(TMP_ICONS_FOLDER, { recursive: true });
};

syncIcons();
