import classnames from 'classnames';
import React, { CSSProperties, FC } from 'react';

import { IClassNameProps } from '@bem-react/core';
import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';

import { Text } from '../Text';
import {
  AddressAddIcon,
  AddressDescriptrion,
  AddressIcon,
  AddressType,
  MAX_LINES_NUMBER,
} from './constants';

import styles from './AddressCard.module.css';

const generateInfo = (info: InfoType): string => {
  const infoParams = [];

  if (info.locality) {
    infoParams.push(info.locality);
  }
  if (info.street) {
    infoParams.push(info.street);
  }
  if (info.building) {
    infoParams.push(`д. ${info.building}`);
  }
  if (info.room) {
    infoParams.push(`кв. ${info.room}`);
  }
  if (info.entrance) {
    infoParams.push(`${info.entrance} подъезд`);
  }
  if (info.floor) {
    infoParams.push(`${info.floor} этаж`);
  }

  return infoParams.join(', ');
};

export interface InfoType {
  locality?: string;
  street?: string;
  building?: number;
  room?: number;
  entrance?: number;
  floor?: number;
}

export interface AddressCardProps extends IClassNameProps, PressProps {
  addressType?: AddressType;
  locationImageUrl?: string;
  info?: InfoType;
  href?: string;
}

export const AddressCard: FC<AddressCardProps> = ({
  addressType = AddressType.Other,
  info,
  locationImageUrl,
  href,
  className,
  ...restProps
}) => {
  const { pressProps } = usePress(restProps);
  const { isHovered, hoverProps } = useHover({ disabled: false });

  const HostElement = (href && 'a') || 'div';

  const style: CSSProperties | undefined = locationImageUrl
    ? {
        backgroundImage: `url("${locationImageUrl}")`,
        backgroundPosition: 'bottom',
      }
    : undefined;

  return (
    <HostElement
      className={classnames(styles.container, className)}
      data-hovered={isHovered}
      href={href}
      {...restProps}
      {...mergeProps(hoverProps, pressProps)}
      style={style}
    >
      <div className={classnames(styles.addressIconContainer)}>
        {locationImageUrl ? AddressIcon[addressType] : AddressAddIcon[addressType]}
      </div>

      <div data-new={!info} className={classnames(styles.description)}>
        {locationImageUrl && (
          <label className={classnames(styles.addressType)}>{addressType}</label>
        )}
        <Text
          data-new={!info}
          className={classnames(styles.info)}
          as="p"
          overflow="ellipsis"
          color="primary"
          maxLines={MAX_LINES_NUMBER}
        >
          {info ? generateInfo(info) : AddressDescriptrion[addressType]}
        </Text>
      </div>
    </HostElement>
  );
};
