import classnames from 'classnames';
import 'focus-visible';
import React, { FC } from 'react';

import { mergeProps, useHover, usePress } from '@use-platform/react';
import { MoreVerticalIcon } from '@yandex-id/icons';
import { UserPic } from '@yandex-lego/components/UserPic';

import { useBankCardRest } from './BankCard.hooks/useBankCardRest';
import { useBankCardState } from './BankCard.hooks/useBankCardState';
import { BankCardProps, BankCardSizes } from './BankCard@types';
import { BankCardSkeleton } from './Skeleton/BankCardSkeleton';
import { BankCardType } from './constants';

import styles from './BankCard.module.css';

interface BankCardDescriptionProps {
  cardNumber?: string;
  label?: string;
  money?: string;
  size?: BankCardSizes;
  bankCardType: BankCardType;
}

const BankCardDescription: FC<BankCardDescriptionProps> = ({
  bankCardType,
  cardNumber,
  label,
  size = 'l',
  money,
}) => {
  let info = cardNumber;
  let subInfo = null;

  if (bankCardType === BankCardType.FamilyMember && money && size === 'l') {
    subInfo = label;
    info = money;
  } else if (label && money) {
    subInfo = label;
    info = money;
  } else if (label) {
    subInfo = label;
    info = cardNumber;
  } else if (money) {
    subInfo = cardNumber;
    info = money;
  }

  return (
    <div className={classnames(styles.description)}>
      {subInfo && (
        <span className={styles.subInfo} data-family={bankCardType === BankCardType.FamilyMember}>
          {subInfo}
        </span>
      )}
      <h4 className={styles.info} data-family={bankCardType === BankCardType.FamilyMember}>
        {info}
      </h4>
    </div>
  );
};

export const BankCard: FC<BankCardProps> = ({ className, size = 'l', ...props }) => {
  const { restProps } = useBankCardRest(props);
  const bankCardData = useBankCardState(
    props.bankCardType !== BankCardType.FamilyMember ? props.bank : undefined,
    size,
    props.bankCardType !== BankCardType.FamilyMember ? props.paymentSystem : undefined,
  );
  const { pressProps } = usePress(props);
  const { isHovered, hoverProps } = useHover({ disabled: props.disabled });

  if (props.bankCardType === BankCardType.FamilyMember && props.isLoading) {
    return (
      <div
        className={classnames(styles.container, styles[`size_${size}`], className)}
        data-loading={props.isLoading}
        {...restProps}
      >
        <BankCardSkeleton />
      </div>
    );
  }

  let style;
  let bankIcon;
  let paymentIcon;

  if (props.bankCardType !== BankCardType.FamilyMember) {
    style = {
      backgroundColor: bankCardData.backgroundColor,
      color: bankCardData.textColor,
    };

    bankIcon = bankCardData.bankIcon;
    paymentIcon = bankCardData.paymentIcon;
  } else {
    bankIcon = <UserPic {...props.userPicProps} />;
  }

  return (
    <div
      className={classnames(
        styles.container,
        props.bankCardType === BankCardType.FamilyMember ? styles.family : '',
        styles[`size_${size}`],
        className,
      )}
      data-hovered={isHovered}
      {...restProps}
      {...mergeProps(hoverProps, pressProps)}
      tabIndex={props.disabled ? -1 : 0}
      style={style}
    >
      <div className={classnames(styles.top)}>
        {bankIcon && <div className={classnames(styles.bankIconContainer)}>{bankIcon}</div>}
        {!props.disabled && (
          <div className={classnames(styles.moreIconContainer)}>
            <MoreVerticalIcon size={16} />
          </div>
        )}
      </div>
      <div className={classnames(styles.bottom)}>
        <BankCardDescription {...props} size={size} />
        {paymentIcon && (
          <div className={classnames(styles.paymentIconContainer)}>{paymentIcon}</div>
        )}
      </div>
    </div>
  );
};
