import { RefObject, useCallback, useEffect, useState } from 'react';

import type { GetElement, GetVisible } from '../Carousel@types';

export interface UseArrowButtonProps {
  /**
   * Ссылка на прокручиваемый элемент.
   */
  scrollableElementRef: RefObject<HTMLElement>;
  /**
   * Метод для определения видимости кнопки со стрелкой.
   */
  getVisible: GetVisible;
  /**
   * Метод для определения элемента (цели прокрутки) после нажатия на кнопку со стрелкой.
   */
  getElement: GetElement;
}

export interface UseArrowButtonResult {
  /**
   * Флаг определяющий видимость кнопки со стрелкой.
   * true - кнопку видно, false - нет.
   */
  isVisible: boolean;
  /**
   * Св-ва кнопки со стрелкой.
   */
  buttonProps: {
    onPress: () => void;
  };
}

export function useArrowButton(props: UseArrowButtonProps): UseArrowButtonResult {
  const { scrollableElementRef, getVisible, getElement } = props;

  const [isVisible, setVisible] = useState(false);

  useEffect(() => {
    const scrollableElement = scrollableElementRef.current;

    if (!scrollableElement) {
      return;
    }

    const changeVisibility = () => {
      setVisible(getVisible(scrollableElement));
    };

    scrollableElement.addEventListener('scroll', changeVisibility);

    // @ts-expect-error
    const { MutationObserver, ResizeObserver } = window;

    const mutationObserver = MutationObserver && new MutationObserver(changeVisibility);
    const resizeObserver = ResizeObserver && new ResizeObserver(changeVisibility);

    mutationObserver?.observe(scrollableElement, {
      childList: true,
    });
    resizeObserver?.observe(scrollableElement);

    changeVisibility();

    return () => {
      scrollableElement.removeEventListener('scroll', changeVisibility);

      mutationObserver?.disconnect();
      resizeObserver?.disconnect();
    };
  }, [scrollableElementRef, getVisible]);

  const onPress = useCallback(() => {
    const scrollableElement = scrollableElementRef.current;

    if (!scrollableElement) {
      return;
    }

    getElement(scrollableElement)?.scrollIntoView({
      inline: 'start',
      block: 'nearest',
      behavior: 'smooth',
    });
  }, [scrollableElementRef, getElement]);

  return {
    isVisible,
    buttonProps: {
      onPress,
    },
  };
}
