import cx from 'classnames';
import React, { Fragment, ReactNode, forwardRef, useRef } from 'react';

import { mergeProps, useForwardedRef, useHover } from '@use-platform/react';

import {
  UseCodeFieldProps,
  UseCodeFieldStateProps,
  useCodeField,
  useCodeFieldState,
} from '../../libs/code-field';
import { Segment } from './Segment';

import styles from './CodeField.module.css';

export interface CodeFieldProps extends UseCodeFieldProps, UseCodeFieldStateProps {
  className?: string;
  type?: string;
  before?: ReactNode;
  after?: ReactNode;
  inputMode?: 'numeric' | 'text';
  invalid?: boolean;
  formatBy?: number;
}

export const CodeField = forwardRef<HTMLDivElement, CodeFieldProps>(function CodeField(
  props,
  forwardedRef,
) {
  const { disabled, before, after, type, inputMode, invalid, formatBy = 3, className } = props;
  const ref = useRef<HTMLDivElement>(null);

  useForwardedRef(ref, forwardedRef);
  const { hoverProps, isHovered } = useHover(props);
  const state = useCodeFieldState(props);
  const { fieldProps } = useCodeField<HTMLDivElement>(props, state);

  return (
    <div
      ref={ref}
      className={cx(
        styles.root,
        {
          [styles.root_isHovered]: isHovered,
          [styles.root_isFocused]: state.focused,
          [styles.root_isDisabled]: disabled,
          [styles.root_isInvalid]: invalid,
        },
        className,
      )}
      {...mergeProps(hoverProps, fieldProps)}
    >
      {before && <div className={styles.slot}>{before}</div>}
      <div className={styles.slot}>
        {state.segments.map((segment, index) => (
          <Fragment key={index}>
            {index > 0 && index % formatBy === 0 && <div className={styles.delimiter} />}
            <Segment
              key={segment.index}
              {...segment}
              type={type}
              inputMode={inputMode}
              disabled={disabled}
              state={state}
              invalid={invalid}
            />
          </Fragment>
        ))}
      </div>
      {after && <div className={styles.slot}>{after}</div>}
    </div>
  );
});
