import React, { useCallback } from 'react';

import { cn } from '@bem-react/classname';
import { IClassNameProps } from '@bem-react/core';

import { useMatchMedia } from '../../utils/hooks';
import { DrawerPhone as Drawer, IDrawerPhoneProps } from '../Drawer';
import { Icon } from '../Icon';
import { IModalProps, Modal } from '../Modal';

import './Dialog.css';

const cnDialog = cn('Dialog');

interface IBaseDialogProps extends IClassNameProps {
  type?: 'modal' | 'drawer' | 'auto';
  classNameContent?: string;
}

export interface IDialogModalProps extends IBaseDialogProps, IModalProps {
  type: 'modal';
  withCloseCross?: boolean;
}
export interface IDialogDrawerProps extends IBaseDialogProps, IDrawerPhoneProps {
  type: 'drawer';
}

export interface IDialogAutoProps extends IBaseDialogProps {
  type: 'auto';
  autoMaxWidth?: number;
  modal: Omit<IDialogModalProps, 'type'>;
  drawer: Omit<IDialogDrawerProps, 'type'>;
}

const DialogDrawer: React.FC<IDialogDrawerProps> = (props) => {
  const { type, classNameContent, className, view = 'default', children, ...restProps } = props;

  return (
    <Drawer {...restProps} view={view} className={cnDialog({ type }, [className])}>
      <div className={cnDialog('Content', { type }, [classNameContent])}>{children}</div>
    </Drawer>
  );
};

const DialogModal: React.FC<IDialogModalProps> = (props) => {
  const {
    type,
    className,
    classNameContent,
    theme = 'normal',
    withCloseCross,
    children,
    ...restProps
  } = props;

  const clickHandler = useCallback((e) => restProps.onClose?.call(null, e, 'esc'), []);

  return (
    <Modal {...restProps} className={cnDialog({ type }, [className])} theme={theme}>
      <div className={cnDialog('Content', { type }, [classNameContent])}>
        {withCloseCross && (
          <div className={cnDialog('Close', { type })} onClick={clickHandler}>
            <Icon glyph="type-close" />
          </div>
        )}
        {children}
      </div>
    </Modal>
  );
};

const DialogAuto: React.FC<IDialogAutoProps> = (props) => {
  const { autoMaxWidth = 768, children, drawer: drawerProps, modal: modalProps } = props;

  const isMatchMaxWidth = useMatchMedia(`(max-width: ${autoMaxWidth}px)`);

  if (isMatchMaxWidth) {
    return (
      <DialogDrawer {...drawerProps} type="drawer">
        {children}
      </DialogDrawer>
    );
  }

  return (
    <DialogModal {...modalProps} type="modal">
      {children}
    </DialogModal>
  );
};

export const Dialog: React.FC<IDialogModalProps | IDialogDrawerProps | IDialogAutoProps> = (
  props,
) => {
  switch (props.type) {
    case 'modal':
      return <DialogModal {...props} />;
    case 'drawer':
      return <DialogDrawer {...props} />;
    default:
      return <DialogAuto {...props}>{props.children}</DialogAuto>;
  }
};
