import 'focus-visible';
import React, { ElementType, FC, ReactNode } from 'react';

import { cn } from '@bem-react/classname';
import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';

import './Hint.css';

export interface HintProps extends PressProps {
  before?: ReactNode | ((isHovered: boolean) => ReactNode);
  after?: ReactNode | ((isHovered: boolean) => ReactNode);
  text: string;
  link?: ReactNode;
  tabIndex?: number;
  href?: string;
  title?: string;
  as?: ElementType;
  target?: string;
  rel?: string;
  to?: string;
  className?: string;
  variant?: 'neutral' | 'positive' | 'negative' | 'info';
}

const b = cn('Hint');

export const Hint: FC<HintProps> = (props) => {
  const { className, variant = 'neutral', before, children, text, after } = props;
  const { pressProps } = usePress(props);
  let disabled = true;
  let tabIndex: number | undefined = undefined;
  let isPressable = false;
  let rel: string | undefined;
  let title: string | undefined;
  let target: string | undefined;
  let to: string | undefined;
  let HostElement: ElementType = 'div';

  if (props.href || props.as) {
    ({ rel, title, target } = props);
    if (props.as) {
      to = props.to;
    }
    HostElement = props.as ?? 'a';
    isPressable = true;
    disabled = false;
  }

  if (
    (['onPress', 'onPressEnd', 'onPressStart', 'onPressUp'] as (keyof PressProps)[]).some(
      (propKey) => typeof props[propKey] === 'function',
    )
  ) {
    isPressable = true;
    disabled = false;
    tabIndex = props.tabIndex ?? 0;
  }

  const { isHovered, hoverProps } = useHover({ disabled });
  const beforeContent = typeof before === 'function' ? before(isHovered) : before;
  const afterContent = typeof after === 'function' ? after(isHovered) : after;

  return (
    <HostElement
      data-pressable={isPressable}
      data-hovered={isHovered}
      data-has-after={Boolean(afterContent)}
      className={b({ variant }, [className])}
      {...mergeProps(hoverProps, pressProps)}
      tabIndex={tabIndex}
      href={props.href}
      target={target}
      rel={rel}
      title={title}
      to={to}
    >
      {beforeContent && <div className={b('left')}>{beforeContent}</div>}
      <div className={b('center')}>
        <div className={b('text')}>{children || text}</div>
        {props.link && <div className={b('link')}>{props.link}</div>}
      </div>
      {afterContent && <div className={b('right')}>{afterContent}</div>}
    </HostElement>
  );
};
