import 'focus-visible';
import React, {
  AnchorHTMLAttributes,
  ElementType,
  ForwardedRef,
  ReactElement,
  forwardRef,
} from 'react';

import { PressProps, mergeProps, useHover, usePress } from '@use-platform/react';

import { createSlot, useSlots } from '../../libs/slots';
import { ElementTypeProps, InferElement } from '../../types/components';
import { ListItemViewSlot, ListItemViewSlotProps } from './Slot';
import { cnListItemView } from './constants';

import './ListItemView.css';

const BeforeSlot = createSlot<ListItemViewSlotProps>('ListItemView.Before');

const AfterSlot = createSlot<ListItemViewSlotProps>('ListItemView.After');

const ContentSlot = createSlot<ListItemViewSlotProps>('ListItemView.Content');

export interface ListItemViewProps<T extends ElementType = 'div'>
  extends AnchorHTMLAttributes<InferElement<T>>,
    PressProps<InferElement<T>> {
  as?: T;
  interactive?: boolean;
  alignItems?: 'top' | 'center';
}

function ListItemView(props: ListItemViewProps<'div'>, ref: ForwardedRef<HTMLDivElement>) {
  const {
    as: Element = 'div',
    className,
    interactive = true,
    alignItems = 'top',
    onPress,
    onPressStart,
    onPressEnd,
    onPressUp,
    ...otherProps
  } = props;
  const { isHovered, hoverProps } = useHover(props);
  const { pressProps } = usePress(props);
  const slots = useSlots(props, { defaultSlot: ContentSlot });

  const before = slots.get(BeforeSlot);
  const after = slots.get(AfterSlot);
  const content = slots.get(ContentSlot);

  return (
    <Element
      ref={ref}
      className={cnListItemView(null, [className])}
      {...mergeProps(otherProps, hoverProps, pressProps)}
      data-interactive={interactive || undefined}
      data-hovered={isHovered || undefined}
    >
      {before && (
        <ListItemViewSlot
          {...before.props}
          alignItems={before.props.alignItems ?? alignItems}
          data-slot="before"
        >
          {before.rendered}
        </ListItemViewSlot>
      )}

      <ListItemViewSlot
        {...content?.props}
        alignItems={content?.props.alignItems ?? alignItems}
        data-slot="content"
      >
        {slots.children}
      </ListItemViewSlot>

      {after && (
        <ListItemViewSlot
          {...after.props}
          alignItems={after.props.alignItems ?? alignItems}
          data-slot="after"
        >
          {after.rendered}
        </ListItemViewSlot>
      )}
    </Element>
  );
}

interface ListItemViewComponent {
  <T extends ElementType = 'div'>(
    props: ElementTypeProps<T, ListItemViewProps<T>>,
  ): ReactElement | null;
}

const _ListItemView = Object.assign(forwardRef(ListItemView) as ListItemViewComponent, {
  Before: BeforeSlot,
  After: AfterSlot,
  Content: ContentSlot,
});

export { _ListItemView as ListItemView };
