import React, { FC, ReactElement, useEffect, useRef, useState } from 'react';

import { Icon, ListItemView, Text } from '@yandex-id/components';
import { Rect, Text as TextSkeleton } from '@yandex-int/tap-components/Skeleton';

interface ExampleDocument {
  name: string;
  icon: string;
  url: string;
}

const documents: ExampleDocument[] = [
  { name: 'Паспорт РФ', icon: 'passport', url: 'https://ya.ru' },
  { name: 'Загранпаспорт', icon: 'passport', url: 'https://ya.ru' },
  { name: 'ИНН', icon: 'inn', url: 'https://ya.ru' },
  { name: 'СНИЛС', icon: 'snils', url: 'https://ya.ru' },
  { name: 'Полис ОМС', icon: 'oms', url: 'https://ya.ru' },
];

const Divider = () => {
  const style = {
    height: '1px',
    margin: '4px 12px 4px 72px',
    backgroundColor: 'var(--id-color-line-normal)',
  };

  return <div style={style} />;
};

const ArrowIcon = () => {
  const style = {
    width: '24px',
    height: '24px',
    display: 'inline-flex',
    justifyContent: 'center',
    alignItems: 'center',
  };

  return (
    <div style={style}>
      <Icon type="arrowRight" />
    </div>
  );
};

interface DocumentProps {
  doc: ExampleDocument;
}

const Document: FC<DocumentProps> = (props) => {
  const { doc } = props;
  const [loading, setLoading] = useState(true);
  const ref = useRef<HTMLAnchorElement>(null);

  useEffect(() => {
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1500 * Math.random());

    return () => {
      clearTimeout(timer);
    };
  }, []);

  return (
    <ListItemView
      ref={ref}
      as="a"
      alignItems="center"
      rel="noreferrer"
      target="_blank"
      href={doc.url}
    >
      <ListItemView.Before>
        {loading ? (
          <Rect className="custom-skeleton" width={44} height={44} borderRadius={12} />
        ) : (
          <Icon type={doc.icon} />
        )}
      </ListItemView.Before>
      <ListItemView.Content>
        {loading ? (
          <TextSkeleton className="custom-skeleton" width="100%" size={20} />
        ) : (
          <Text variant="text-m" color="primary">
            {doc.name}
          </Text>
        )}
      </ListItemView.Content>

      <ListItemView.After>
        <ArrowIcon />
      </ListItemView.After>
    </ListItemView>
  );
};

const skeletonStyles = `
.custom-skeleton {
    --skeleton-color: var(--id-color-decor-neutral-100);
}
`;

export const Links = () => {
  const content: ReactElement[] = [];

  for (let i = 0; i < documents.length; i++) {
    content.push(<Document key={i} doc={documents[i]} />);

    if (i !== documents.length - 1) {
      content.push(<Divider key={`divider-${i}`} />);
    }
  }

  return (
    <div>
      <style>{skeletonStyles}</style>
      {content}
    </div>
  );
};
