import React from 'react';

import { cn } from '@bem-react/classname';

import { INavLink } from '../../types';
import { isLinkActive } from '../../utils';
import { PageLink } from '../Link';

import './Navigation.css';

interface IPageNavigationProps {
  navigationLinks: INavLink[];
  otherNavigationLinks?: INavLink[];
  specialNavigationLinks?: INavLink[];
  pathname: string;
  aside?: boolean;
  className?: string;
  linkComponent?: React.ReactType;
}

const b = cn('PageNavigation');

const renderLink = (
  link: INavLink,
  as: IPageNavigationProps['linkComponent'],
  aside: IPageNavigationProps['aside'],
  activeLink?: INavLink,
) => {
  return (
    <li key={link.href || link.to} className={b('listItem')}>
      {!link.separator && (
        <PageLink
          to={link.to}
          href={link.href}
          blank={link.blank}
          as={as}
          className={b('link', {
            row: !aside,
            col: aside,
            active: (activeLink?.to || activeLink?.href) === (link.to || link.href),
          })}
        >
          {aside && link.icon && <span className={b('linkIcon')}>{link.icon}</span>}
          <span className={b('linkTitle')}>{link.title}</span>
        </PageLink>
      )}
    </li>
  );
};

export const PageNavigation: React.FC<IPageNavigationProps> = ({
  aside,
  className,
  pathname,
  linkComponent,
  navigationLinks,
  specialNavigationLinks,
  otherNavigationLinks,
}) => {
  const activeLink =
    [...navigationLinks, ...(specialNavigationLinks || [])].find(
      ({ to, href }, index) => (to || href) && index > 0 && isLinkActive(to || href, pathname),
    ) || navigationLinks[0];

  return (
    <nav className={b({ aside }, [className])}>
      <ul className={b('list')}>
        {navigationLinks.map((link) => renderLink(link, linkComponent, aside, activeLink))}
        {specialNavigationLinks && specialNavigationLinks.length > 0 && (
          <>
            <li className={b('listItem', { sep: true })} />
            {specialNavigationLinks.map((link) =>
              renderLink(link, linkComponent, aside, activeLink),
            )}
          </>
        )}
        {otherNavigationLinks?.length && (
          <>
            <li className={b('listItem', { sep: true })} />
            {otherNavigationLinks.map((link) => renderLink(link, linkComponent, aside, activeLink))}
          </>
        )}
      </ul>
    </nav>
  );
};
