import cx from 'classnames';
import React, { HTMLAttributes, forwardRef, useMemo } from 'react';

import { UseProgressBarProps, useProgressBar } from '../../libs/progressbar';
import { calcDashOffset, calcPercentage, calcStrokeWidth, clamp, makeCirclePath } from './utils';

import styles from './ProgressCircle.module.css';

// NOTE: Используется для корректного масштабирования текста и иконки
const TEXT_BASE_SIZE = 44;
// NOTE: Радиус, при котором stroke-dasharray = 360.
// Нужен для того, чтобы удобно работать css-анимацией
const RADIUS = 57.29577951308232;
const CIRCUMFERENS = 2 * Math.PI * RADIUS;

export interface ProgressCircleProps extends UseProgressBarProps, HTMLAttributes<HTMLSpanElement> {
  variant?: 'default' | 'brand';
  size?: 16 | 24 | 32 | 36 | 40 | 44;
  clockwise?: boolean;
  strokeWidth?: number;
  strokeLinecap?: 'round' | 'butt';
}

export const ProgressCircle = forwardRef<HTMLSpanElement, ProgressCircleProps>((props, ref) => {
  const {
    value: valueProp = 0,
    min = 0,
    max = 100,
    variant = 'default',
    size = 36,
    clockwise = true,
    formatStyle,
    strokeWidth: strokeWidthProp = 2,
    strokeLinecap = 'round',
    indeterminate,
    className,
    ...otherProps
  } = props;
  const { progressBarProps } = useProgressBar(props);
  const value = clamp(valueProp, min, max);
  const percentage = calcPercentage(value, min, max);
  const strokeWidth = calcStrokeWidth(RADIUS, size, strokeWidthProp);
  const viewBoxSize = 2 * RADIUS + strokeWidth;
  const center = viewBoxSize / 2;
  const dashOffset = calcDashOffset(CIRCUMFERENS, percentage, clockwise);
  const transformScale = `scale(${viewBoxSize / TEXT_BASE_SIZE})`;

  const circlePath = useMemo(() => {
    return makeCirclePath(center, center, RADIUS, clockwise);
  }, [center, clockwise]);

  const formattedText = useMemo(() => {
    if (formatStyle === 'percent') {
      return `${Math.round(percentage * 100)}%`;
    }

    if (formatStyle === 'decimal') {
      return `${value}/${max}`;
    }

    return null;
  }, [value, max, percentage, formatStyle]);

  return (
    <span
      ref={ref}
      className={cx(styles.root, className)}
      {...otherProps}
      {...progressBarProps}
      data-variant={variant}
      data-indeterminate={indeterminate || undefined}
      data-percentage={indeterminate ? undefined : percentage}
      data-clockwise={clockwise || undefined}
    >
      <svg
        className={styles.svg}
        width={size}
        height={size}
        viewBox={`0 0 ${viewBoxSize} ${viewBoxSize}`}
      >
        <circle
          className={styles.track}
          cx={center}
          cy={center}
          r={RADIUS}
          strokeWidth={strokeWidth}
        />

        <path
          className={styles.dash}
          d={circlePath}
          strokeWidth={strokeWidth}
          strokeLinecap={strokeLinecap}
          strokeDasharray={CIRCUMFERENS}
          strokeDashoffset={dashOffset}
        />

        {formattedText && (
          <>
            {percentage < 1 && (
              <text
                className={styles.text}
                x={TEXT_BASE_SIZE / 2}
                y={TEXT_BASE_SIZE / 2}
                dominantBaseline="center"
                transform={transformScale}
              >
                {formattedText}
              </text>
            )}

            {percentage === 1 && (
              <path
                className={styles.completeIcon}
                d="M14.4999 22L18.7479 27.0976C19.1423 27.5708 19.8666 27.5784 20.2707 27.1136L29.4999 16.5"
                fill="none"
                strokeWidth="3"
                strokeLinecap="round"
                strokeLinejoin="round"
                transform={transformScale}
              />
            )}
          </>
        )}
      </svg>
    </span>
  );
});
