import cx from 'classnames';
import React, { CSSProperties, FC, HTMLAttributes, PropsWithChildren } from 'react';

import { cn } from '@bem-react/classname';

import { assert } from '../../libs/assert/assert';
import { createSlot, useSlots } from '../../libs/slots';

import './PromoContent.css';

export interface PromoContentProps {
  align?: 'center' | 'start';
  className?: string;
}

export interface PromoContentControlProps {
  align?: 'start' | 'center';
  marginTop?: number;
}

const b = cn('PromoContent');

const promoContentClassName = b();
const promoContentImageClassName = b('Image');
const promoContentSupertitleClassName = b('Supertitle');
const promoContentTitleClassName = b('Title');
const promoContentDescriptionClassName = b('Description');
const promoContentControlClassName = b('Control');
const promoContentButtonClassName = b('Button');
const promoContentSecondaryButtonClassName = b('SecondaryButton');

const Title = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentTitle');
const Button = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentButton');
const Supertitle = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentSupertitle');
const Image = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentImage');
const Description = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentDescription');
const SecondaryButton = createSlot<HTMLAttributes<HTMLDivElement>>('PromoContentSecondaryButton');
const Control = createSlot<PromoContentControlProps>('PromoContentControl');

interface InternalControlProps extends HTMLAttributes<HTMLDivElement> {
  align?: 'start' | 'center';
  marginTop?: number;
}

const InternalControl: FC<InternalControlProps> = (props) => {
  const { align, children, marginTop, ...otherProps } = props;
  const style: CSSProperties = { marginTop };

  return (
    <div
      {...otherProps}
      data-align={align}
      style={style}
      className={cx(promoContentControlClassName, otherProps.className)}
    >
      {children}
    </div>
  );
};

const _PromoContent: FC<PropsWithChildren<PromoContentProps>> = (props) => {
  const slotMap = useSlots(props);

  const titleSlot = slotMap.get(Title);
  const buttonSlot = slotMap.get(Button);
  const descriptionSlot = slotMap.get(Description);
  const imageSlot = slotMap.get(Image);
  const supertitleSlot = slotMap.get(Supertitle);
  const secondaryButtonSlot = slotMap.get(SecondaryButton);
  const controlSlot = slotMap.get(Control);

  assert(titleSlot, () => {
    const missingSlots: string[] = [];

    if (!titleSlot) {
      missingSlots.push('PromoContentTitle');
    }

    return `Required slot${missingSlots.length > 1 ? 's' : ''} ${missingSlots.join(
      ' and ',
    )} was not found!`;
  });

  return (
    <div className={cx(promoContentClassName, props.className)} data-align={props.align}>
      {imageSlot && (
        <div
          {...imageSlot.props}
          className={cx(promoContentImageClassName, imageSlot.props.className)}
        >
          {imageSlot.rendered}
        </div>
      )}
      {supertitleSlot && (
        <div
          {...supertitleSlot.props}
          className={cx(promoContentSupertitleClassName, supertitleSlot.props.className)}
        >
          {supertitleSlot.rendered}
        </div>
      )}
      <div
        {...titleSlot.props}
        className={cx(promoContentTitleClassName, titleSlot.props.className)}
      >
        {titleSlot.rendered}
      </div>
      {descriptionSlot && (
        <div
          {...descriptionSlot.props}
          className={cx(promoContentDescriptionClassName, descriptionSlot.props.className)}
        >
          {descriptionSlot.rendered}
        </div>
      )}
      {controlSlot && (
        <InternalControl {...controlSlot.props} align={controlSlot.props.align ?? props.align}>
          {controlSlot.rendered}
        </InternalControl>
      )}
      {buttonSlot && (
        <div
          {...buttonSlot.props}
          className={cx(promoContentButtonClassName, buttonSlot.props.className)}
        >
          {buttonSlot.rendered}
        </div>
      )}
      {secondaryButtonSlot && (
        <div
          {...secondaryButtonSlot.props}
          className={cx(promoContentSecondaryButtonClassName, secondaryButtonSlot.props.className)}
        >
          {secondaryButtonSlot.rendered}
        </div>
      )}
    </div>
  );
};

export const PromoContent = Object.assign(_PromoContent, {
  Title,
  Button,
  Supertitle,
  Image,
  Description,
  SecondaryButton,
  Control,
});
