import classNames from 'classnames';
import 'focus-visible';
import React, { ElementType, FC, ReactNode, useRef } from 'react';

import { IClassNameProps } from '@bem-react/core';
import { ButtonBaseProps, mergeProps, useButton, useHover } from '@use-platform/react';

import { ShortcutButton } from './Button/ShortcutButton';
import { TextGroup, TextGroupProps } from './TextGroup/TextGroup';

import styles from './Shortcut.module.css';

export interface ShortcutProps extends TextGroupProps, IClassNameProps, ButtonBaseProps {
  withImage?: boolean;
  icon?: ReactNode;
  as?: ElementType;
  disabled?: boolean;
  href?: string;
  target?: string;
  referrerPolicy?: HTMLAnchorElement['referrerPolicy'];
  to?: string;
  buttonText?: string;
  amount?: string;
  amountIcon?: ReactNode;
  currency?: string;
}

export const ShortcutComponent: FC<ShortcutProps> = ({
  withImage,
  icon,
  as,
  className,
  buttonText,
  text,
  textBold,
  label,
  labelBold,
  to,
  href,
  amount,
  amountIcon,
  currency,
  target,
  referrerPolicy,
  ...otherProps
}) => {
  const shortcutRef = useRef<HTMLButtonElement>(null);
  const { isHovered, hoverProps } = useHover(otherProps);
  const { buttonProps } = useButton(otherProps, shortcutRef);
  const HostElement = as || (href && 'a') || 'div';

  const Wrapper = icon && (buttonText || amount) && (text || label) ? 'div' : React.Fragment;

  return (
    <HostElement
      data-hovered={isHovered}
      data-pressable={!otherProps.disabled}
      to={to}
      href={href}
      className={classNames(styles.container, className)}
      target={target}
      referrerPolicy={referrerPolicy}
      ref={shortcutRef}
      {...mergeProps(otherProps, hoverProps, buttonProps)}
    >
      {withImage && <div className={styles.image} />}

      <Wrapper>
        {icon && <div className={styles.icon}>{icon}</div>}

        <TextGroup label={label} text={text} textBold={textBold} labelBold={labelBold} />
      </Wrapper>

      {amount && (
        <div className={styles.amount}>
          {amountIcon && <div className={styles.amountIcon}>{amountIcon}</div>}
          <div
            className={classNames(styles.amountText, {
              [styles.amountTextM]: amount.length === 3,
              [styles.amountTextS]: amount.length >= 4,
            })}
          >
            {amount}
          </div>
          {currency && <div className={styles.amountCurrency}>{currency}</div>}
        </div>
      )}

      {buttonText && <ShortcutButton>{buttonText}</ShortcutButton>}
    </HostElement>
  );
};
