import cx from 'classnames';
import React, { HTMLAttributes, forwardRef, useContext, useMemo, useRef } from 'react';

import { useForwardedRef } from '@use-platform/react';
import { ThemeContext, useTheme } from '@yandex-id/design-system';

import { createSlot, useSlots } from '../../libs/slots';
import { Progress } from './Progress';

import styles from './Snackbar.module.css';

const BeforeSlot = createSlot<HTMLAttributes<HTMLDivElement>>('Snackbar.Before');
const ContentSlot = createSlot<HTMLAttributes<HTMLDivElement>>('Snackbar.Content');
const AfterSlot = createSlot<HTMLAttributes<HTMLDivElement>>('Snackbar.After');
const ProgressSlot = createSlot<HTMLAttributes<HTMLDivElement>>('Snackbar.Progress');

export interface SnackbarProps extends HTMLAttributes<HTMLDivElement> {
  colorScheme?: 'light' | 'dark';
  duration?: number;
  onClose?: () => void;
}

const _Snackbar = forwardRef<HTMLDivElement, SnackbarProps>(function Snackbar(props, forwardedRef) {
  const { className, colorScheme = 'dark', duration, onClose, ...restProps } = props;
  const { themeProps } = useTheme({ colorScheme });
  const parentThemeContext = useContext(ThemeContext);
  const slots = useSlots(props, { defaultSlot: ContentSlot });
  const ref = useRef<HTMLDivElement>(null);

  const before = slots.get(BeforeSlot);
  const after = slots.get(AfterSlot);
  const content = slots.get(ContentSlot);
  const progress = slots.get(ProgressSlot);

  useForwardedRef(ref, forwardedRef);

  const themeContext = useMemo(() => {
    return { ...parentThemeContext, colorScheme };
  }, [parentThemeContext, colorScheme]);

  return (
    <div ref={ref} className={cx(styles.root, className)} {...restProps}>
      <ThemeContext.Provider value={themeContext}>
        <div className={styles.container} {...themeProps}>
          <Progress {...progress?.props} duration={duration} onClose={onClose} />

          <div className={styles.inner}>
            {before && (
              <div {...before.props} className={cx(styles.before, before.props.className)}>
                {before.rendered}
              </div>
            )}

            <div {...content?.props} className={cx(styles.content, content?.props.className)}>
              {slots.children}
            </div>
          </div>

          {after && (
            <div {...after.props} className={cx(styles.after, after.props.className)}>
              {after.rendered}
            </div>
          )}
        </div>
      </ThemeContext.Provider>
    </div>
  );
});

export const Snackbar = Object.assign(_Snackbar, {
  Before: BeforeSlot,
  After: AfterSlot,
  Content: ContentSlot,
  Progress: ProgressSlot,
});
