import React from 'react';

import { Story } from '@storybook/react';
import { ProgressCircle, Snackbar, SnackbarProps, Text } from '@yandex-id/components';
import { Button } from '@yandex-id/components/next';

import { CheckIcon, CloseIcon, ErrorIcon } from './icons';

const BEFORE_CONTENT = {
  none: null,
  loading: <ProgressCircle variant="brand" size={24} indeterminate />,
  success: (
    <Text variant="text-m" style={{ display: 'inline-flex' }}>
      <CheckIcon />
    </Text>
  ),
  danger: (
    <Text variant="text-m" color="negative" style={{ display: 'inline-flex' }}>
      <ErrorIcon />
    </Text>
  ),
};

const AFTER_CONTENT = {
  none: null,
  icon: (
    <Button variant="clear" size="m">
      <CloseIcon />
    </Button>
  ),
  button: (
    <Button variant="clear" size="m">
      Закрыть
    </Button>
  ),
  'button-icon': (
    <>
      <Button variant="clear" size="m">
        Повторить
      </Button>
      <Button variant="clear" size="m">
        <CloseIcon />
      </Button>
    </>
  ),
};

interface DefaultArgs extends Omit<SnackbarProps, 'duration'> {
  animation: boolean;
  before: 'none' | 'loading' | 'success' | 'danger';
  after: 'none' | 'button' | 'button-icon';
  duration?: string;
}

export const Default: Story<DefaultArgs> = (args) => {
  const {
    children,
    before = 'none',
    after = 'none',
    duration: durationProp = 'Infinity',
    ...otherArgs
  } = args;
  const beforeContent = BEFORE_CONTENT[before];
  const afterContent = AFTER_CONTENT[after];

  const duration = parseInt(durationProp.toString(), 10);

  return (
    <div
      data-testid="safe-area"
      style={{ padding: '24px', display: 'flex', justifyContent: 'center' }}
    >
      <Snackbar duration={duration} {...otherArgs}>
        {beforeContent && <Snackbar.Before>{beforeContent}</Snackbar.Before>}
        <Text variant="text-m">{children}</Text>
        {afterContent && <Snackbar.After>{afterContent}</Snackbar.After>}
      </Snackbar>
    </div>
  );
};

Default.args = {
  size: 'm',
  width: 'auto',
  children: 'Yandex ID',
  before: 'none',
  after: 'none',
  colorScheme: 'dark',
  duration: 'Infinity',
};

Default.argTypes = {
  onClose: {
    control: {
      type: 'action',
    },
  },
  size: {
    control: {
      type: 'select',
    },
    options: ['m', 'l'],
  },
  width: {
    control: {
      type: 'select',
    },
    options: ['auto', 'available'],
  },
  colorScheme: {
    control: {
      type: 'select',
    },
    options: ['dark', 'light'],
  },
  before: {
    control: {
      type: 'select',
    },
    options: ['none', 'loading', 'success', 'danger'],
  },
  after: {
    control: {
      type: 'select',
    },
    options: ['none', 'icon', 'button', 'button-icon'],
  },
  duration: {
    control: {
      type: 'select',
    },
    options: ['Infinity', '5000', '10000'],
  },
};
