import cx from 'classnames';
import React, { VFC, useCallback, useMemo, useState } from 'react';
import { createPortal } from 'react-dom';
import { useSyncExternalStore } from 'use-sync-external-store/shim';

import { useIsomorphicLayoutEffect as useLayoutEffect } from '@use-platform/react/libs/isomorphic-layout-effect';

import { SnackbarItem } from './SnackbarItem';
import { SnackbarStore } from './SnackbarStore';
import { SnackbarPosition } from './types';

import styles from './SnackbarContainer.module.css';

export interface SnackbarContainerProps {
  store: SnackbarStore;
  gap?: number;
  position: SnackbarPosition;
  maxCount?: number;
  reverseOrder?: boolean;
  className?: string;
}

export const SnackbarContainer: VFC<SnackbarContainerProps> = (props) => {
  const {
    className,
    position = 'bottom-center',
    gap = 8,
    reverseOrder,
    maxCount = 1,
    store,
  } = props;

  const snapshot = useSyncExternalStore(store.subscribe, store.getQueue);
  const snackbars = useMemo(() => {
    const count = Math.max(1, maxCount);

    return snapshot.slice(count * -1);
  }, [maxCount, snapshot]);
  const ids = useMemo(() => snackbars.map((s) => s.id), [snackbars]);
  const [heightMap, setHeightMap] = useState<Record<string, number | undefined>>({});
  const [offsets, setOffsets] = useState<Record<string, number | undefined>>({});

  const handleUpdateHeight = useCallback(
    (id: string, height: number) => {
      setHeightMap((map) => {
        const newValue = ids.reduce((acc, itemId) => {
          acc[itemId] = map[itemId];

          return acc;
        }, {} as Record<string, number | undefined>);

        newValue[id] = height;

        return newValue;
      });
    },
    [ids],
  );

  useLayoutEffect(() => {
    const dir = position.includes('top') ? 1 : -1;

    setOffsets(() => {
      const newValue: Record<string, number | undefined> = {};

      ids.forEach((id, index) => {
        const relevantIds = reverseOrder ? ids.slice(index + 1) : ids.slice(0, index);
        const offset = relevantIds.reduce((acc, itemId) => acc + (heightMap[itemId] || 0) + gap, 0);

        newValue[id] = offset * dir;
      });

      return newValue;
    });
  }, [ids, position, reverseOrder, gap, heightMap]);

  if (snackbars.length === 0 || typeof window === 'undefined') {
    return null;
  }

  return createPortal(
    <div className={cx(styles.root, className)}>
      {snackbars.map((snackbar) => {
        const { id, component, visible } = snackbar;
        const offset = offsets[id] || 0;

        return (
          <SnackbarItem
            key={id}
            id={id}
            component={component}
            offset={offset}
            visible={visible}
            position={position}
            onClose={store.hide}
            onChangeVisibility={store.updateVisibility}
            onUpdateHeight={handleUpdateHeight}
          />
        );
      })}
    </div>,
    document.body,
  );
};
