import React, {
  AnimationEvent,
  CSSProperties,
  ComponentType,
  VFC,
  useCallback,
  useMemo,
  useRef,
} from 'react';

import { useIsomorphicLayoutEffect as useLayoutEffect } from '@use-platform/react/libs/isomorphic-layout-effect';

import { SnackbarComponentProps, SnackbarPosition } from '../types';

import styles from './SnackbarItem.module.css';

export interface SnackbarItemProps {
  id: string;
  visible: boolean;
  position: SnackbarPosition;
  offset: number;
  component: ComponentType<SnackbarComponentProps>;
  onClose: (id: string) => void;
  onChangeVisibility: (id: string, visible: boolean) => void;
  onUpdateHeight: (id: string, height: number) => void;
}

export const SnackbarItem: VFC<SnackbarItemProps> = (props) => {
  const {
    id,
    component: Component,
    visible,
    position,
    offset,
    onClose,
    onChangeVisibility,
    onUpdateHeight,
  } = props;
  const contentRef = useRef<HTMLDivElement>(null);
  const [primary, secondary] = position.split('-');

  const style = useMemo(() => {
    const style: CSSProperties = {
      transform: `translateY(${offset}px)`,
    };

    return style;
  }, [offset]);

  const handleAnimationEnd = useCallback(
    (event: AnimationEvent<HTMLDivElement>) => {
      if (event.target === event.currentTarget) {
        const isVisible = event.currentTarget.dataset.visible === 'true';

        event.currentTarget.dataset.visibility = isVisible ? 'visible' : 'hidden';

        onChangeVisibility(id, isVisible);
      }
    },
    [id, onChangeVisibility],
  );

  const handleClose = useCallback(() => {
    onClose(id);
  }, [id, onClose]);

  useLayoutEffect(() => {
    const element = contentRef.current;

    if (!element) {
      return;
    }

    const { animationName } = getComputedStyle(element);

    if (animationName === 'none') {
      element.dataset.visibility = visible ? 'visible' : 'hidden';

      onChangeVisibility(id, visible);
    }
  }, [id, visible, onChangeVisibility]);

  useLayoutEffect(() => {
    if (contentRef.current) {
      const rect = contentRef.current.getBoundingClientRect();

      onUpdateHeight(id, rect.height);
    }
  }, [id, onUpdateHeight]);

  return (
    <div
      className={styles.root}
      data-primary-position={primary}
      data-secondary-position={secondary}
      style={style}
    >
      <div
        ref={contentRef}
        className={styles.content}
        data-visible={visible}
        onAnimationEnd={handleAnimationEnd}
      >
        <Component id={id} onClose={handleClose} />
      </div>
    </div>
  );
};
