import { ComponentType } from 'react';

import { SnackbarComponentProps, SnackbarState } from './types';

export class SnackbarStore {
  protected serial = 0;

  protected queue: SnackbarState[] = [];

  protected subscribers: Set<() => void> = new Set();

  constructor() {
    this.getQueue = this.getQueue.bind(this);
    this.subscribe = this.subscribe.bind(this);
    this.show = this.show.bind(this);
    this.hide = this.hide.bind(this);
    this.update = this.update.bind(this);
    this.updateVisibility = this.updateVisibility.bind(this);
  }

  protected notify() {
    this.subscribers.forEach((fn) => {
      fn();
    });
  }

  show(component: ComponentType<SnackbarComponentProps>) {
    const id = `snackbar-${this.serial++}`;

    this.queue = [
      {
        id,
        component,
        visible: true,
        visibility: 'hidden',
      },
      ...this.queue,
    ];

    this.notify();

    return id;
  }

  hide(id: string) {
    const snackbar = this.queue.find((snack) => snack.id === id);

    if (!snackbar) {
      return;
    }

    this.queue = this.queue.reduce<SnackbarState[]>((acc, snack) => {
      if (snack.id !== id) {
        acc.push(snack);
      } else if (snack.visibility !== 'hidden') {
        acc.push({ ...snack, visible: false });
      }

      return acc;
    }, []);

    this.notify();
  }

  clear() {
    if (this.queue.length === 0) {
      return;
    }

    this.queue = this.queue.reduce<SnackbarState[]>((acc, snack) => {
      if (snack.visibility !== 'hidden') {
        acc.push({ ...snack, visible: false });
      }

      return acc;
    }, []);

    this.notify();
  }

  update(id: string, component: ComponentType<SnackbarComponentProps>) {
    this.queue = this.queue.map((snack) => {
      return snack.id === id ? { ...snack, component } : snack;
    });

    this.notify();
  }

  subscribe(fn: () => void) {
    let isSubscribed = true;

    this.subscribers.add(fn);

    return () => {
      if (!isSubscribed) {
        return;
      }

      isSubscribed = false;

      this.subscribers.delete(fn);
    };
  }

  getQueue() {
    return this.queue;
  }

  updateVisibility(id: string, visible: boolean) {
    this.queue = this.queue.map((snack) => {
      if (snack.id === id) {
        const visibility = visible ? 'visible' : 'hidden';

        return { ...snack, visibility };
      }

      return snack;
    });

    this.queue = this.queue.filter((item) => item.visible || item.visibility === 'visible');

    this.notify();
  }
}
