import React, {
  CSSProperties,
  ComponentType,
  ElementType,
  ForwardedRef,
  HTMLAttributes,
  ReactElement,
  forwardRef,
} from 'react';

import { cn } from '@bem-react/classname';

import { ElementTypeProps, InferElement } from '../../types/components';

import './Text.css';

const cnText = cn('id-Text');

export type TextVariant =
  | 'display-l'
  | 'display-m'
  | 'display-s'
  | 'heading-xl'
  | 'heading-l'
  | 'heading-m'
  | 'heading-s'
  | 'text-l'
  | 'text-l-long'
  | 'text-m'
  | 'text-m-long'
  | 'text-s'
  | 'text-s-long'
  | 'text-xs'
  | 'text-xs-long'
  | 'text-xxs';

export type TextColor = 'inherit' | 'primary' | 'secondary' | 'tertiary' | 'negative' | 'positive';

export type TextWeight = 'regular' | 'medium' | 'bold';

export interface TextProps<T extends ElementType = 'span'> extends HTMLAttributes<InferElement<T>> {
  as?: T;
  variant?: TextVariant;
  color?: TextColor;
  weight?: TextWeight;
  overflow?: 'ellipsis';
  maxLines?: number;
}

function Text<T extends ElementType>(props: TextProps<T>, ref: ForwardedRef<InferElement<T>>) {
  const {
    as = 'span',
    maxLines = 1,
    variant,
    color = 'primary',
    weight,
    overflow,
    children,
    className,
    style: styleProp,
    ...otherProps
  } = props;
  const Element = as as ComponentType<ElementTypeProps<T, TextProps<T>>>;
  const style: CSSProperties = { ...styleProp };

  if (overflow === 'ellipsis') {
    style.WebkitLineClamp = maxLines;
  }

  return (
    <Element
      ref={ref}
      className={cnText(null, [className])}
      style={style}
      {...otherProps}
      data-variant={variant}
      data-weight={weight}
      data-color={color}
      data-overflow={overflow}
    >
      {children}
    </Element>
  );
}

interface TextComponent {
  <T extends ElementType = 'span'>(props: ElementTypeProps<T, TextProps<T>>): ReactElement | null;
}

const _Text = forwardRef(Text) as TextComponent;

export { _Text as Text };
