import React from 'react';
import replace from 'react-string-replace';

import { cn } from '@bem-react/classname';
import { IClassNameProps } from '@bem-react/core';

import { AppStore } from './Logo/AppStore';
import { GooglePlay } from './Logo/GooglePlay';
import { PhonePromo } from './Logo/PhonePromo';
import { Ya } from './Logo/Ya';

import './YandexAppPromo.css';

export interface IYandexAppPromoProps extends IClassNameProps {
  device?: 'android' | 'ios' | 'desktop';
  lang: string;
  title: string;
  installMessage: string;
  description: string;
  usrImg: string;
}

export const cnYandexAppPromo = cn('YandexAppPromo');

const APP_STORE_LINK =
  'https://redirect.appmetrica.yandex.com/serve/386347106450828605?c=passport_bage';
const GOOGLE_PLAY_LINK =
  'https://redirect.appmetrica.yandex.com/serve/458404701881276453?c=passport_bage';

const reg = /\{([a-z|A-Z|0-9|.]+)\}/g;

const getDictionaryReplacements = (lang: string, usrImg: string): Record<string, JSX.Element> => ({
  break: <br />,
  logo: <Ya className={cnYandexAppPromo('YaLogo')} lang={lang} />,
  avatar: <img className={cnYandexAppPromo('UserImg')} src={usrImg} alt="" />,
});

const getStoreLanguage = (lang: string) => {
  switch (lang) {
    case 'ru':
    case 'ua':
    case 'kz':
    case 'by':
      return 'ru';
    default:
      return 'en';
  }
};

export const YandexAppPromo: React.FC<IYandexAppPromoProps> = ({
  className,
  device = 'desktop',
  description,
  title,
  installMessage,
  usrImg,
  lang,
}) => {
  const isMobile = Boolean(device === 'android' || device === 'ios');
  const storeLang = getStoreLanguage(lang);

  const dict = getDictionaryReplacements(lang, usrImg);

  const titleFormatted = replace(title, reg, (match) => dict[match]);
  const descriptionFormatted = replace(description, reg, (match) => dict[match]);
  const messageFormatted = replace(installMessage, reg, (match) => dict[match]);

  return (
    <div className={cnYandexAppPromo(null, [className])}>
      <PhonePromo className={cnYandexAppPromo('Logo')} full={!isMobile} />
      <div className={cnYandexAppPromo('Info')}>
        <div className={cnYandexAppPromo('Message', { upper: true })}>{titleFormatted}</div>
        <div className={cnYandexAppPromo('Message', { lower: true })}>{descriptionFormatted}</div>
        <div className={cnYandexAppPromo('Install')}>
          <span className={cnYandexAppPromo('InstallText')}>{messageFormatted}</span>
          <div className={cnYandexAppPromo('InstallButtons')}>
            {(device === 'android' || device === 'desktop') && (
              <div className={cnYandexAppPromo('InstallAndroid')}>
                <a href={GOOGLE_PLAY_LINK} aria-label="Google Play">
                  <GooglePlay lang={storeLang} />
                </a>
              </div>
            )}
            {(device === 'ios' || device === 'desktop') && (
              <div className={cnYandexAppPromo('InstallIos')}>
                <a href={APP_STORE_LINK} aria-label="App Store">
                  <AppStore lang={storeLang} />
                </a>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};
