import {
  ChangeEvent,
  InputHTMLAttributes,
  KeyboardEvent,
  RefObject,
  useCallback,
  useState,
} from 'react';

import { mergeProps, useFocus } from '@use-platform/react';
import { useIsomorphicLayoutEffect } from '@use-platform/react/libs/isomorphic-layout-effect';

import { UseCodeFieldStateResult } from './use-code-field-state';

export interface UseCodeFieldInputProps {
  type?: string;
  index: number;
  value: string;
  selected?: boolean;
  disabled?: boolean;
  inputMode?: 'numeric' | 'text';
}

export function useCodeFieldInput(
  props: UseCodeFieldInputProps,
  state: UseCodeFieldStateResult,
  ref: RefObject<HTMLInputElement>,
) {
  const { type = 'text', value, selected, disabled, inputMode } = props;
  const { update, focused: fieldFocused } = state;
  const [focused, setFocused] = useState(false);
  const { focusProps } = useFocus({ onFocusChange: setFocused });

  useIsomorphicLayoutEffect(() => {
    const isFocused = focused && document.activeElement === ref.current;

    if (ref.current && fieldFocused && selected && !isFocused) {
      ref.current.focus();
    }
  }, [ref, fieldFocused, focused, selected]);

  const handleKeyDown = useCallback(
    (event: KeyboardEvent<HTMLInputElement>) => {
      switch (event.key) {
        case 'Backspace':
          event.preventDefault();
          update('');
          break;
      }
    },
    [update],
  );

  const handleChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      update(event.target.value);
    },
    [update],
  );

  const inputProps: InputHTMLAttributes<HTMLInputElement> = {
    type,
    value,
    size: 1,
    disabled,
    autoComplete: 'off',
    inputMode: inputMode || type === 'number' ? 'numeric' : 'text',
    onChange: handleChange,
    onKeyDown: handleKeyDown,
  };

  if (!disabled && !selected) {
    inputProps.tabIndex = -1;
  }

  return {
    inputProps: mergeProps(inputProps, focusProps),
  };
}
