import { useCallback, useEffect, useMemo, useRef, useState } from 'react';

export interface CodeSegment {
  index: number;
  value: string;
  selected: boolean;
}

export interface UseCodeFieldStateProps {
  value?: string;
  codeLength?: number;
  disabled?: boolean;
  onChange?: (code: string) => void;
  onComplete?: (code: string) => void;
}

export interface UseCodeFieldStateResult {
  value: string;
  segments: CodeSegment[];
  focused: boolean;
  setFocused: (value: boolean) => void;
  update: (value: string) => void;
}

export function useCodeFieldState(props: UseCodeFieldStateProps): UseCodeFieldStateResult {
  const { value = '', codeLength: propCodeLength = 6, disabled, onChange, onComplete } = props;
  // NOTE: Коды у нас могут быть либо 4, либо 6 символов, но на всякий случай разрешаем указать
  // длину в диапазоне от 1 до 12
  const codeLength = Math.min(Math.max(propCodeLength, 1), 12);
  const selectedIndex = value.length < codeLength ? value.length : codeLength - 1;
  const [focused, setFocused] = useState(false);
  const prevRef = useRef<string | undefined>();

  useEffect(() => {
    if (value.length > 0 && prevRef.current !== value && value.length === codeLength) {
      onComplete?.(value);
    }

    prevRef.current = value;
  }, [codeLength, value, onComplete]);

  const segments = useMemo(() => {
    return Array.from({ length: codeLength }, (_, index) => {
      const segment: CodeSegment = {
        index,
        value: value.charAt(index),
        selected: !disabled && selectedIndex === index,
      };

      return segment;
    });
  }, [codeLength, value, disabled, selectedIndex]);

  const setValue = useCallback(
    (code: string) => {
      if (value === code) {
        return;
      }

      onChange?.(code);
    },
    [value, onChange],
  );

  const update = useCallback(
    (segmentValue: string) => {
      if (disabled) {
        return;
      }

      if (segmentValue.length === 0) {
        setValue(value.substring(0, value.length - 1));

        return;
      }

      // TODO: Сделать более универсальное решение
      // сейчас это необходимо, чтобы коды `123-456` и `123 456` правильно обрабатывались
      // Но валидацию кода нужно пока делать вне компонента
      const striped = segmentValue.replace(/[^0-9a-zA-Z]/g, '');

      if (striped.length === 0) {
        return;
      }

      const start = value.substring(0, selectedIndex);
      const end = value.substring(selectedIndex + striped.length);

      const newValue = `${start}${striped}${end}`.substring(0, codeLength);

      setValue(newValue);
    },
    [disabled, value, selectedIndex, codeLength, setValue],
  );

  return {
    value,
    segments,
    focused,
    setFocused,
    update,
  };
}
