import { HTMLAttributes, useEffect, useRef } from 'react';

import { mergeProps, useFocusWithin } from '@use-platform/react';

import { UseCodeFieldStateResult } from './use-code-field-state';

export interface UseCodeFieldProps {
  disabled?: boolean;
  autoFocus?: boolean;
}

export interface UseCodeFieldResult<T = HTMLElement> {
  fieldProps: HTMLAttributes<T>;
}

export function useCodeField<T extends HTMLElement = HTMLElement>(
  props: UseCodeFieldProps,
  state: UseCodeFieldStateResult,
): UseCodeFieldResult<T> {
  const { disabled, autoFocus } = props;
  const { setFocused } = state;
  const { focusWithinProps } = useFocusWithin<T>({
    disabled,
    onFocusWithinChange: state.setFocused,
  });
  const autoFocused = useRef(Boolean(disabled));

  useEffect(() => {
    if (!autoFocused.current && autoFocus) {
      autoFocused.current = true;

      setFocused(true);
    }
  }, [autoFocus, setFocused]);

  const fieldProps: HTMLAttributes<T> = {
    // NOTE: делаем фокусируемым, чтобы при клике сработал onFocusWithinChange
    tabIndex: -1,
    'aria-disabled': disabled ? true : undefined,
  };

  return {
    fieldProps: mergeProps(fieldProps, focusWithinProps),
  };
}
