/* eslint-disable valid-jsdoc */
import React, {
  ComponentType,
  NamedExoticComponent,
  PropsWithChildren,
  ReactNode,
  RefAttributes,
  forwardRef,
} from 'react';

import type { StylesChunk, Variants } from './types';
import { mergeStyles } from './utils/merge-styles';
import { mergeVariants } from './utils/merge-variants';

export interface Options<T, D> {
  variants?: T;
  displayName?: string;
  defaultProps?: D;
  styles?: StylesChunk | StylesChunk[];
}

export interface DefaultProps<T> {
  defaultProps: T;
}

export interface SteelyExoticComponent<P, V> extends NamedExoticComponent<P> {
  __options?: Options<V, any>;
  __component?: SteelyComponentType<any, any>;
}

type SteelyComponentType<T, R> =
  | keyof JSX.IntrinsicElements
  | ComponentType<T>
  | SteelyExoticComponent<T, R>;

type VariantProps<T> = {
  [K in keyof T]: T[K] extends Record<string, any> ? keyof T[K] : boolean;
};

export type WithVariantProps<P, V, R> =
  // Remove previous variant props from current and add extended variant props.
  Omit<P, keyof R> &
    VariantProps<V & R> &
    RefAttributes<any> &
    PropsWithChildren<{ className?: string; as?: ReactNode }>;

/**
 * Creates or extends component with styles and variants.
 *
 * Generic types:
 * P — props
 * V — variants
 * R — prev variants
 * D — defaultProps
 */
export function component<P, V extends Variants = {}, R extends Variants = {}, D = {}>(
  ComponentType: SteelyComponentType<P, R>,
  options: Options<V, D>,
): SteelyExoticComponent<WithVariantProps<P, V, R>, V & R> & DefaultProps<D> {
  const Component = (ComponentType as any).__component ?? ComponentType;
  const prevOptions = (ComponentType as any).__options ?? {};

  const variants = mergeVariants(prevOptions.variants, options.variants);
  const styles = mergeStyles(prevOptions.styles, options.styles);

  const displayName = options.displayName ?? (ComponentType as any).displayName ?? 'Component';

  const NextComponent = forwardRef((props: any, ref: any) => {
    const nextStyles = [...styles];
    const nextProps = { ...props };

    // TOOD: Возможно стоит кэшировать данный участок кода.
    for (const key of Object.keys(variants)) {
      if (nextProps[key] === undefined) {
        continue;
      }

      let variantCase: StylesChunk | null = null;

      if (typeof variants[key] === 'object') {
        variantCase = (variants[key] as Record<string, StylesChunk>)[nextProps[key]];
      } else {
        variantCase = nextProps[key] ? (variants[key] as StylesChunk) : null;
      }

      delete nextProps[key];

      if (!variantCase) {
        continue;
        // throw new Error(`Variant case not found ${key}: ${nextProps[key]} for ${displayName}`);
      }

      if (Array.isArray(variantCase)) {
        nextStyles.push(...variantCase);
      } else {
        nextStyles.push(variantCase);
      }
    }

    const className = [nextProps.className, ...nextStyles].filter(Boolean).join(' ');

    return <Component {...nextProps} ref={ref} className={className} />;
  });

  NextComponent.displayName = `Steely(${displayName})`;
  NextComponent.defaultProps = options.defaultProps;

  // @ts-expect-error
  NextComponent.__options = { styles, variants };
  // @ts-expect-error
  NextComponent.__component = Component;

  return NextComponent as any;
}
