import { HTMLAttributes, useMemo } from 'react';

import { useNumberFormatter } from '../number-formatter';

export interface UseProgressBarProps {
  /**
   * Minimum progress value.
   *
   * @default 0
   */
  min?: number;

  /**
   * Maximum progress value.
   *
   * @default 100
   */
  max?: number;

  /**
   * Current progress value.
   *
   * @default 0
   */
  value?: number;

  /**
   * Indeterminate state, when progress is unknown.
   */
  indeterminate?: boolean;

  /**
   * Display format of the value label.
   *
   * @default 'percent'
   */
  formatStyle?: 'percent' | 'decimal';
}

export interface UseProgressBarResult {
  progressBarProps: HTMLAttributes<HTMLElement>;
}

// TODO: Move to @use-platform/react
export function useProgressBar(props: UseProgressBarProps): UseProgressBarResult {
  const { value = 0, min = 0, max = 100, indeterminate, formatStyle = 'percent' } = props;
  const clampedValue = Math.min(Math.max(value, min), max);
  const formatOptions = useMemo<Intl.NumberFormatOptions>(() => {
    return { style: formatStyle };
  }, [formatStyle]);
  const formatter = useNumberFormatter(formatOptions);
  const percentage = (clampedValue - min) / (max - min);
  const valueToFormat = formatStyle === 'percent' ? percentage : clampedValue;
  const formattedValue = useMemo(() => {
    return formatter.format(valueToFormat);
  }, [formatter, valueToFormat]);

  const progressBarProps: HTMLAttributes<HTMLElement> = {
    role: 'progressbar',
    'aria-valuemin': min,
    'aria-valuemax': max,
  };

  if (!indeterminate) {
    progressBarProps['aria-valuenow'] = clampedValue;
    progressBarProps['aria-valuetext'] = formattedValue;
  }

  return {
    progressBarProps,
  };
}
