import cx from 'classnames';
import 'focus-visible';
import React, {
  ButtonHTMLAttributes,
  ForwardedRef,
  ReactNode,
  forwardRef,
  isValidElement,
  useRef,
} from 'react';

import {
  PressEventProps,
  mergeProps,
  useButton,
  useForwardedRef,
  useHover,
} from '@use-platform/react';

import { CustomizableProps } from '../../libs/css-modules';

import styles from './Button.module.css';

export interface ButtonProps extends CustomizableProps, PressEventProps {
  type?: ButtonHTMLAttributes<HTMLButtonElement>['type'];
  after?: ReactNode;
  before?: ReactNode;
  children?: ReactNode;
  disabled?: boolean;
  href?: string;
  tabIndex?: number;
  target?: string;
  rel?: string;
  as?: React.ElementType;
}

function _Button(props: ButtonProps, ref: ForwardedRef<HTMLButtonElement>) {
  const { className, before, after, children, as: ElementType = 'button', ...otherProps } = props;
  const elementType = typeof ElementType === 'string' ? ElementType : 'button';

  const buttonRef = useRef<HTMLButtonElement>(null);
  const { buttonProps, isPressed } = useButton({ ...props, elementType }, buttonRef);
  const { hoverProps, isHovered } = useHover(props);

  useForwardedRef(buttonRef, ref);

  const isCustomChildren = isValidElement(children);

  return (
    <ElementType
      {...mergeProps(otherProps, buttonProps, hoverProps)}
      className={cx(
        styles.root,
        {
          [styles.root_withoutGap]: isCustomChildren,
          [styles.root_isPressed]: isPressed,
          [styles.root_isHovered]: isHovered,
        },
        className,
      )}
      ref={buttonRef}
    >
      {before && (
        <span data-addon="before" className={styles.addon}>
          {before}
        </span>
      )}
      {isCustomChildren ? children : <span className={styles.text}>{children}</span>}
      {after && (
        <span data-addon="after" className={styles.addon}>
          {after}
        </span>
      )}
    </ElementType>
  );
}

export const Button = forwardRef(_Button);
