import cx from 'classnames';
import 'focus-visible';
import React, { ChangeEvent, FC, useCallback, useRef, useState } from 'react';

import {
  CommonCheckboxProps,
  mergeProps,
  useCheckbox,
  useFocusWithin,
  useHover,
} from '@use-platform/react';

import { CustomizableProps } from '../../libs/css-modules';

import styles from './Checkbox.module.css';

export interface CheckboxProps extends CommonCheckboxProps, CustomizableProps {
  onChangeValue?: (isChecked: boolean) => void;
}

export const Checkbox: FC<CheckboxProps> = (props) => {
  const { children, className, onChange, onChangeValue, ...otherProps } = props;
  const { checked: isChecked } = props;

  const inputRef = useRef<HTMLInputElement>(null);
  const [isFocused, setFocused] = useState(false);

  const handleOnChange = useCallback(
    (event: ChangeEvent<HTMLInputElement>) => {
      onChange?.(event);
      onChangeValue?.(event.target.checked);
    },
    [onChange, onChangeValue],
  );

  const onFocusWithinChange = useCallback(() => {
    setFocused(inputRef.current?.dataset.focusVisibleAdded !== undefined);
  }, []);

  const { isPressed, inputProps, rootProps } = useCheckbox(
    { ...otherProps, onChange: handleOnChange },
    inputRef,
  );
  const { isHovered, hoverProps } = useHover(otherProps);
  const { focusWithinProps } = useFocusWithin({ onFocusWithinChange });

  return (
    <label
      className={cx(
        styles.root,
        {
          [styles.root_isChecked]: isChecked,
          [styles.root_isFocused]: isFocused,
          [styles.root_isHovered]: isHovered,
          [styles.root_isPressed]: isPressed,
        },
        className,
      )}
      {...mergeProps(rootProps, hoverProps, focusWithinProps)}
    >
      <input ref={inputRef} className={styles.input} {...inputProps} />
      <span className={styles.box}>
        <svg
          aria-hidden
          className={styles.tick}
          fill="none"
          focusable={false}
          role="img"
          strokeDasharray={26}
          strokeDashoffset={26}
          viewBox="0 0 24 24"
        >
          <path
            d="m4 12 5.5 6L20 6"
            stroke="currentColor"
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={4}
          />
        </svg>
      </span>
      {children && <span className={styles.label}>{children}</span>}
    </label>
  );
};
