import { request } from '../libs/request';
import { responseAdapter } from './response-adapter';
import { NormalizedSuggestSection, SearchResponse } from './types';

interface FetchOptions {
  signal: AbortSignal;
}

export async function fetchSuggestions(
  path: string,
  options: FetchOptions,
): Promise<NormalizedSuggestSection[]> {
  const { signal } = options;
  const response = await request<SearchResponse>(path, { signal });

  if (response.canceled) {
    return [];
  }

  return responseAdapter(response.data);
}

interface SearchByFilterOptions extends FetchOptions {
  searchText?: string;
}

export async function search(
  path: string,
  options: SearchByFilterOptions,
): Promise<NormalizedSuggestSection[]> {
  const { signal, searchText } = options;
  const response = await request<SearchResponse>(path, {
    signal,
    query: {
      text: searchText,
    },
  });

  if (response.canceled) {
    return [];
  }

  return responseAdapter(response.data);
}

interface ApiOptions {
  suggestUrl: string;
  searchUrl: string;
}

export function createApi(options: ApiOptions) {
  return {
    prefetch: (params: FetchOptions) => {
      return fetchSuggestions(options.suggestUrl, params);
    },
    fetch: (params: SearchByFilterOptions) => {
      return search(options.searchUrl, params);
    },
  };
}
