import { canUseDom } from '@use-platform/react/libs/dom-utils';

import type { OkResponse, RequestOptions } from './types';

const defaultOptions: RequestOptions = {
  method: 'GET',
};

/**
 * Creates request and returns response with data and status.
 */
export async function request<T = unknown>(
  path: string,
  options: RequestOptions = defaultOptions,
): Promise<OkResponse<T>> {
  const { method, body, signal, query } = options;
  const resource = path + getQueryString(query);

  try {
    const response = await fetch(resource, {
      signal,
      method: method,
      body: JSON.stringify(body),
      headers: {
        'Content-Type': 'application/json',
        // @ts-expect-error (Remove this and use config for order-history and user-id)
        'x-csrf-token': canUseDom ? window.__CSRF__ : '',
      },
    });

    if (!response.ok) {
      throw response;
    }

    const result = await response.json();

    return {
      type: 'success',
      status: response.status,
      data: result,
    } as OkResponse<T>;
  } catch (error: any) {
    if (error.name === 'AbortError') {
      return { type: 'success', canceled: true } as OkResponse<T>;
    }

    throw error;
  }
}

function getQueryString(params?: RequestOptions['query']) {
  if (!params) {
    return '';
  }

  const query = new URLSearchParams(params);

  return '?' + query.toString();
}
