import React, {
  FC,
  FocusEvent,
  KeyboardEvent,
  ReactElement,
  useCallback,
  useMemo,
  useRef,
  useState,
} from 'react';
import { useComboBox } from 'react-aria';

// eslint-disable-next-line import/no-extraneous-dependencies
import { ComboBoxProps } from '@react-types/combobox';
// eslint-disable-next-line import/no-extraneous-dependencies
import { FocusStrategy } from '@react-types/shared';

import { Informer } from './components/Informer';
import { ListBox } from './components/ListBox';
import { ListBoxSkeleton } from './components/ListBoxSkeleton';
import { Popover } from './components/Popover';
import { ProtectedOverlay } from './components/ProtectedOverlay';
import { SearchField } from './components/SearchField';
import { ListKeyboardDelegate as Delegate } from './libs/ListKeyboardDelegate';
import { useComboBoxState } from './libs/useComboBoxState';
import type { LayoutProps } from './types';

import './SearchSuggest.css';

export interface SearchSuggestProps<T = {}> extends ComboBoxProps<T>, LayoutProps {
  isError: boolean;
  isLoading: boolean;
}

export const SearchSuggestBase: FC<SearchSuggestProps> = (props) => {
  const { isLoading, isError, layout = 'static', scale = 'medium', ...otherProps } = props;

  const isFullScreenLayout = layout === 'fullscreen';
  const [isFullScreen, setFullScreen] = useState(false);

  const targetRef = useRef<HTMLDivElement>(null);
  const inputRef = useRef<HTMLInputElement>(null);
  const popoverRef = useRef<HTMLDivElement>(null);
  const listBoxRef = useRef<HTMLDivElement>(null);

  const state = useComboBoxState({
    ...otherProps,
    allowsEmptyCollection: true,
    menuTrigger: 'manual',
  });
  const delegate = useMemo(() => new Delegate(state.collection), [state.collection]);
  const { inputProps, listBoxProps } = useComboBox(
    {
      ...otherProps,
      inputRef,
      keyboardDelegate: delegate,
      listBoxRef,
      popoverRef,
      shouldFocusWrap: true,
    },
    { ...state, focusStrategy: false as unknown as FocusStrategy },
  );

  const onClose = useCallback(() => {
    setFullScreen(false);
    state.close();
    state.commit();
  }, [state]);

  const onFocus = useCallback(
    (event: FocusEvent<HTMLInputElement>) => {
      if (isFullScreenLayout) {
        setFullScreen(true);
      }

      inputProps.onFocus?.(event);
      state.open();
    },
    [inputProps, isFullScreenLayout, state],
  );

  const onBlur = useCallback(() => {
    if (!isFullScreenLayout) {
      state.close();
    }
  }, [isFullScreenLayout, state]);

  const onKeyDown = useCallback(
    (event: KeyboardEvent<HTMLInputElement>) => {
      if (event.key === 'Tab') {
        return;
      }
      if (event.key !== 'Enter' || state.selectionManager.focusedKey !== null) {
        inputProps.onKeyDown?.(event);
      }
    },
    [inputProps, state.selectionManager.focusedKey],
  );

  const isEmptyCollection = state.collection.size === 0;

  let content: ReactElement;

  if (isLoading) {
    content = <ListBoxSkeleton />;
  } else if (isError) {
    content = <Informer kind="broken" />;
  } else if (isEmptyCollection) {
    content = <Informer kind="not-found" />;
  } else {
    content = <ListBox {...listBoxProps} ref={listBoxRef} state={state} />;
  }

  return (
    <>
      <div className="ID-Suggest" data-fullscreen={isFullScreen || undefined} data-scale={scale}>
        <SearchField
          {...inputProps}
          innerRef={targetRef}
          onBlur={onBlur}
          onClose={onClose}
          onFocus={onFocus}
          onKeyDown={onKeyDown}
          placeholder="Найти данные и настройки"
          ref={inputRef}
          showCloseButton={isFullScreen}
        />
        <Popover
          isVisible={state.isOpen}
          targetRef={targetRef}
          shouldPreventScroll={isFullScreen}
          ref={popoverRef}
          offset={2}
        >
          {content}
        </Popover>
      </div>
      <ProtectedOverlay isVisible={isFullScreen} />
    </>
  );
};
