// NOTE: In the future this component can be decomposed for each sources.
import React, { FC, useRef } from 'react';
import { useOption } from 'react-aria';
import type { ListState } from 'react-stately';

import { useDateFormatter } from '@use-platform/react';

import type { NormalizedSuggestItem } from '../../../../shared/api';
import {
  Comment,
  CreditCard,
  File,
  Folder,
  Id,
  Key,
  Link,
  Login,
  Mail,
  Passport,
  Phone,
} from '../../../../shared/icons';
import type { ItemProps } from '../../../../shared/libs/collections';
import { Currency, symbols } from '../../../../shared/libs/currency-symbols';

interface ListOptionProps {
  option: ItemProps<NormalizedSuggestItem>;
  state: ListState<unknown>;
}

export const ListOption: FC<ListOptionProps> = (props) => {
  const { state, option } = props;
  const ref = useRef<HTMLDivElement>(null);
  const { optionProps, isPressed, isFocused } = useOption(option, state, ref);
  const dateFormatter = useDateFormatter({ year: 'numeric', month: 'long', day: 'numeric' });

  const date = option.props.date ? dateFormatter.format(option.props.date) : null;
  const icon = getIcon(option.props.source, option.props.scope);
  const description = getDescription([date, option.props.description]);
  const payment = getPayment(option.props.payment);

  return (
    <div
      {...optionProps}
      className="ID-ListBox-Item"
      data-focused={isPressed || isFocused || undefined}
      ref={ref}
    >
      <span className="ID-ListBox-ItemIcon">{icon}</span>
      <span className="ID-ListBox-ItemText">{option.rendered}</span>
      {payment}
      {description}
    </div>
  );
};

const icons: Record<string, FC<{ size?: 12 | 16 | 24 | 32 }>> = {
  // suggest
  suggest$passport: Passport,
  suggest$login: Login,
  suggest$phone: Phone,
  suggest$password: Key,
  suggest$payment: CreditCard,
  suggest$support: Comment,
  // disk
  disk$folder: Folder,
  disk$name: File,
  disk$body: File,
  // mail
  mail: Mail,
  // passport
  passport: Id,
};

function getIcon(source: string, scope?: string) {
  const key = scope ? `${source}$${scope}` : source;
  const ServiceIcon = icons[key] ?? Link;

  return <ServiceIcon size={16} />;
}

function getDescription(chunks: Array<string | undefined | null>) {
  const content = chunks.filter(Boolean).join(' • ');

  if (!content) {
    return null;
  }

  return (
    <span className="ID-ListBox-ItemDescription" title={content}>
      &ensp;—&ensp;
      {content}
    </span>
  );
}

function getPayment(payment: NormalizedSuggestItem['payment']) {
  if (!payment) {
    return null;
  }

  const content = [];
  const currency = symbols[payment.currency as Currency] ?? payment.currency;

  content.push(
    <span className="ID-ListBox-ItemPaymentAmount" key="PaymentAmount">
      &ensp;−{payment.amount}
      {currency}
    </span>,
  );

  if (payment.cashbackAmount) {
    content.push(
      <span className="ID-ListBox-ItemPaymentPlus" key="PaymentPlus">
        &ensp;−{payment.cashbackAmount}
      </span>,
    );
  }

  return <span className="ID-ListBox-ItemPayment">{content}</span>;
}
