import React, { RefObject, forwardRef, useRef } from 'react';

// eslint-disable-next-line import/no-extraneous-dependencies
import { useHover } from '@react-aria/interactions';
import {
  FocusEventProps,
  InputValueProps,
  KeyboardEventProps,
  mergeProps,
  useClearButton,
  useForwardedRef,
  usePress,
  useTextField,
} from '@use-platform/react';

import { ArrowLongBack, Close, Search } from '../../../../shared/icons';
import { FieldButton } from '../FieldButton';

import './SearchField.css';

interface SearchFieldProps
  extends InputValueProps<any>,
    FocusEventProps<HTMLInputElement>,
    KeyboardEventProps<HTMLInputElement> {
  innerRef?: RefObject<HTMLDivElement>;
  onClose?: () => void;
  placeholder?: string;
  showCloseButton?: boolean;
}

export const SearchField = forwardRef<HTMLInputElement, SearchFieldProps>((props, forwardedRef) => {
  const { innerRef, onClose, placeholder, showCloseButton, ...otherProps } = props;
  const ref = useRef<HTMLInputElement>(null);

  useForwardedRef(ref, forwardedRef);

  // TODO: Use hook from web-platform-alpha after fix for touch.
  //       https://github.com/use-platform/use-platform/issues/140
  const { isHovered, hoverProps } = useHover({});
  const { inputProps } = useTextField(otherProps, ref);
  const { isActive, buttonProps } = useClearButton(otherProps, ref);
  const { pressProps } = usePress({
    onPress: () => {
      ref.current?.focus();
    },
  });

  return (
    <div
      {...mergeProps(hoverProps, pressProps)}
      className="ID-SearchField"
      data-hovered={isHovered || undefined}
      ref={innerRef}
    >
      <div className="ID-SearchField-Wrapper">
        {showCloseButton ? (
          <FieldButton onPress={onClose}>
            <ArrowLongBack />
          </FieldButton>
        ) : (
          <Search />
        )}
        <input
          {...inputProps}
          ref={ref}
          type="search"
          className="ID-SearchField-Control"
          placeholder={placeholder}
        />
        {isActive && (
          <FieldButton {...buttonProps}>
            <Close />
          </FieldButton>
        )}
      </div>
    </div>
  );
});
