import { Reducer, useEffect, useReducer, useRef } from 'react';
import { useDebouncedCallback } from 'use-debounce';

export interface UseSuggestListProps<T> {
  prefetch: (options: { signal: AbortSignal }) => Promise<T>;
  fetch: (options: { searchText?: string; signal: AbortSignal }) => Promise<T>;
}

export interface UseSuggestListResult<T> {
  searchText: string;
  isError: boolean;
  isLoading: boolean;
  items: T;
  prefetch: () => void;
  setSearchText: (searchText: string) => void;
}

interface CallFetcherOptions<T> {
  fx: (options: { searchText?: string; signal: AbortSignal }) => Promise<T>;
  isPrefetch?: boolean;
  searchText?: string;
}

const initialState: ReducerState<any> = {
  status: 'idle',
  searchText: '',
  items: [],
  prefetchItems: [],
};

export function useSuggestList<T extends unknown[]>(
  props: UseSuggestListProps<T>,
): UseSuggestListResult<T> {
  type ReducerType = Reducer<ReducerState<T>, ReducerAction<T>>;

  const { prefetch, fetch } = props;

  const [state, dispatch] = useReducer<ReducerType>(reducer, initialState);
  const abortControllerRef = useRef<AbortController | null>(null);
  const callFetcherFx = useDebouncedCallback(_callFetcher, 500, { leading: true });

  async function _callFetcher(options: CallFetcherOptions<T>) {
    const { searchText, fx, isPrefetch } = options;
    const abortController = new AbortController();
    const signal = abortController.signal;

    if (abortControllerRef.current) {
      abortControllerRef.current.abort();
    }

    abortControllerRef.current = abortController;

    try {
      dispatch({ type: 'fetching' });
      const items = await fx({ searchText, signal });

      if (!signal.aborted) {
        // eslint-disable-next-line no-warning-comments
        // FIXME: Find better solution for separated dispatch.
        if (isPrefetch) {
          dispatch({ type: 'prefetch-success', items });
        } else {
          dispatch({ type: 'success', items });
        }
      }
    } catch (error) {
      dispatch({ type: 'failure' });
    }
  }

  useEffect(() => {
    return () => {
      if (abortControllerRef.current) {
        abortControllerRef.current.abort();
        abortControllerRef.current = null;
      }
    };
  }, []);

  return {
    searchText: state.searchText,
    isError: state.status === 'failure',
    isLoading: state.status === 'typing' || state.status === 'loading',
    items: state.searchText ? state.items : state.prefetchItems,

    prefetch: () => {
      if (state.prefetchItems.length === 0) {
        callFetcherFx({ fx: prefetch, isPrefetch: true });
      }
    },

    setSearchText: (searchText: string) => {
      if (searchText) {
        dispatch({ type: 'typing', searchText });
        callFetcherFx({ fx: fetch, searchText });
      } else {
        dispatch({ type: 'idle' });
      }
    },
  };
}

interface ReducerState<T extends unknown[]> {
  items: T;
  searchText: string;
  status: 'idle' | 'typing' | 'loading' | 'success' | 'failure';
  prefetchItems: T;
}

type ReducerAction<T> =
  | { type: 'idle' }
  | { type: 'fetching' }
  | { type: 'typing'; searchText: string }
  | { type: 'success'; items: T }
  | { type: 'prefetch-success'; items: T }
  | { type: 'failure' };

function reducer<T extends unknown[]>(
  state: ReducerState<T>,
  action: ReducerAction<T>,
): ReducerState<T> {
  switch (action.type) {
    case 'idle':
      return {
        ...state,
        status: 'idle',
        searchText: '',
      };
    case 'fetching':
      return {
        ...state,
        status: 'loading',
      };
    case 'typing':
      return {
        ...state,
        status: 'typing',
        searchText: action.searchText,
      };
    case 'prefetch-success':
      return {
        ...state,
        status: 'success',
        prefetchItems: action.items,
      };
    case 'success':
      return {
        ...state,
        status: 'success',
        items: action.items,
      };
    case 'failure':
      return {
        ...state,
        status: 'failure',
      };
  }
}
