import * as fs from 'fs';

import { Address4, Address6 } from 'ip-address';
import type { BigInteger } from 'jsbn';

interface RawNetData {
  low: string;
  high: string;
  flags?: {
    sub?: boolean;
    user?: boolean;
    'u-sub'?: boolean;
    't-sub'?: boolean;
    turbo?: boolean;
    'trusted-proxy': boolean;
  };
}

function isIPV6(a: string) {
  return a.indexOf(':') >= 0;
}

function getAddressInteger(address: string) {
  const ipV = isIPV6(address) ? Address6 : Address4;

  return new ipV(address).bigInteger();
}

class Net {
  readonly isIPv6: boolean;
  readonly interval: string;
  protected readonly start: BigInteger;
  protected readonly end: BigInteger;

  constructor(src: RawNetData) {
    this.isIPv6 = isIPV6(src.low);

    this.start = getAddressInteger(src.low);
    this.end = getAddressInteger(src.high);
    this.interval = `${src.low} - ${src.high} (${this.start} - ${this.end})`;
  }

  toString() {
    return this.interval;
  }

  toJSON() {
    return this.interval;
  }

  [Symbol.toPrimitive]() {
    return this.interval;
  }

  [Symbol.toStringTag]() {
    return this.interval;
  }

  /**
   * Компаратор подсети с адресом или другой подсетью.
   * Для сравнения "подсеть — подсеть" возвращает:
   *  -1 — Если стартовый адрес подсети меньше чем конечный адрес другой подсети
   *   0 —
   *  +1 —
   *
   * Для сравнения "подсеть — адрес" возвращает:
   *  -1 — Если адрес меньше чем стартовый адрес подсети
   *   0 — Если адрес больше чем стартовый адрес и меньше чем конечный адрес подсети
   *  +1 — Если адрес больше чем конечный адрес подсети
   * */
  compareTo(other: Net): -1 | 0 | 1;
  compareTo(other: string | BigInteger): -1 | 0 | 1;
  compareTo(other: Net | string | BigInteger) {
    if (other instanceof Net) {
      if (this.end.compareTo(other.start) < 0) {
        return -1;
      }
      if (this.start.compareTo(other.end) > 0) {
        return +1;
      }
      if (this.start.compareTo(other.start) === 0 && this.end.compareTo(other.end) === 0) {
        return 0;
      }

      throw new Error(`Networks are intersected: ${this.interval} and ${other.interval}`);
    }

    let addrNumber = other;

    if (typeof addrNumber === 'string') {
      addrNumber = getAddressInteger(addrNumber);
    }

    if (this.start.compareTo(addrNumber) > 0) {
      return -1;
    }
    if (this.end.compareTo(addrNumber) < 0) {
      return 1;
    }

    return 0;
  }

  // Алгоритм
  // https://a.yandex-team.ru/arc/trunk/arcadia/geobase/libipreg1/include/ipreg/lookup_impl.hpp?rev=r2986645#L104
  static isUserNet(src: RawNetData) {
    return src?.flags?.user && !src?.flags?.['u-sub'];
  }

  static isYandexNet(src: RawNetData) {
    return !src?.flags?.sub;
  }
}

export class IpregController {
  private userNetList: Net[];
  private yandexNetList: Net[];

  constructor(filePath: string) {
    const rawNetLits = JSON.parse(fs.readFileSync(filePath, 'utf8')) as RawNetData[];

    this.userNetList = [];
    this.yandexNetList = [];

    rawNetLits.forEach((rawNet) => {
      const isUserNet = Net.isUserNet(rawNet);
      const isYandexNet = Net.isYandexNet(rawNet);

      if (!isUserNet && !isYandexNet) {
        return;
      }

      const net = new Net(rawNet);

      if (isUserNet) {
        this.userNetList.push(net);
      }
      if (isYandexNet) {
        this.yandexNetList.push(net);
      }
    });

    this.userNetList.sort((a, b) => a.compareTo(b));
    this.yandexNetList.sort((a, b) => a.compareTo(b));
  }

  get yandexNets() {
    return this.yandexNetList;
  }

  get userNets() {
    return this.userNetList;
  }

  private findNetwork(list: Net[], address: string) {
    const addr = getAddressInteger(address);

    let s = 0;
    let e = list.length;

    while (s < e) {
      const m = Math.floor((e + s) / 2);
      const sNet = list[m];
      const order = sNet.compareTo(addr);

      if (order < 0) {
        e = m;
      } else if (order > 0) {
        s = m + 1;
      } else {
        return sNet;
      }
    }

    return undefined;
  }

  isUserAddr(addr: string): boolean {
    return Boolean(this.findNetwork(this.userNets, addr));
  }

  isYandexAddr(addr: string): boolean {
    return Boolean(this.findNetwork(this.yandexNets, addr));
  }
}
