import { sync as fg } from 'fast-glob';

import { Command, flags } from '@oclif/command';
import { FAILURE_BEHAVIOUR, exec } from '@yandex-int/frontend.ci.utils';
import { getBaseCommitInTrunk } from '@yandex-int/frontend.ci.utils';
import { Affected as AffectedIver } from '@yandex-lego/iver';
import { AffectedPackageType } from '@yandex-lego/iver/lib/affected/typings';

import { getVcs } from '../vcs';

export default class Affected extends Command {
  vcs = getVcs(this.log);

  static description = 'Команда для запуска npm скриптов на затронутых пакетах';

  static examples = ['$ monorepo affected ci:lint --master'];

  static flags = {
    help: flags.help({ char: 'h' }),
    all: flags.boolean({ description: 'Запустить на все узлы' }),
    linkDependant: flags.boolean({ description: 'Линковать зависимости при запуске' }),
    master: flags.boolean({
      description: 'Запуск на мастере (чтобы захватить изменения последнего ПР)',
    }),
  };

  static args = [
    {
      name: 'command',
      description: 'Название команды, которую необходимо запустить',
      required: true,
    },
  ];

  getLernaFilter(packages: AffectedPackageType[]) {
    const packageNames = packages.map((_package) => `--scope ${_package.name}`);

    return packageNames.join(' ');
  }

  async getAffected(master: boolean) {
    let since: string | undefined = undefined;

    if (master) {
      since = getBaseCommitInTrunk();
    }

    const command = new AffectedIver({ vcs: { since } });
    const result = await command.execute();

    return result.sortedAffectedPackages;
  }

  async run() {
    let EXIT_CODE = 0;

    const { args, flags } = this.parse(Affected);
    const command = args.command;
    const affected = await this.getAffected(flags.master);

    if (!flags.all && affected.length === 0) {
      this.log('Нет затронутых пакетов');

      return;
    }

    let filter = this.getLernaFilter(affected);

    if (flags.linkDependant) {
      filter += ' --include-filtered-dependents';
    }

    if (flags.all) {
      filter = '';
    }

    const run = command === 'bootstrap' ? 'bootstrap' : `run ${command}`;
    const cmd = `lerna ${run} ${filter}`;

    this.beforeCommand(command);

    this.log('Запускаю команду:');
    this.log(cmd);

    try {
      exec(cmd, FAILURE_BEHAVIOUR.THROW, process.cwd(), true);
    } catch (e) {
      console.error(e);
      EXIT_CODE = 1;
    }

    this.log('Выполнение команды завершено');
    this.afterCommand(command);
    process.exit(EXIT_CODE);
  }

  beforeCommand(command: string) {
    switch (command) {
      case 'ci:lint':
        // Чтобы селективно запустить линтеры, надо сбросить изменения до мастера и поместить их в staging
        this.vcs.resetToTrunk();
    }
  }
  afterCommand(command: string) {
    switch (command) {
      case 'ci:lint':
        // После lint-staged восстанавливаем рабочую копию
        this.vcs.returnToBrach();
        this.publishReport();
    }
  }
  publishReport() {
    const reports = fg(['**/report-lint-*']);

    for (const report of reports) {
      exec(`trendbox publish-report ${report}`);
    }
  }
}
