import { Command, flags } from '@oclif/command';
import { exec, getBaseCommitInTrunk } from '@yandex-int/frontend.ci.utils';
import { Affected as AffectedIver, SeverityEnum, Version } from '@yandex-lego/iver';

import { ENV } from '../env';
import { getVcs } from '../vcs';

export default class Publish extends Command {
  vcs = getVcs(this.log);
  static description = 'Команда для публикации измененных пакетов';

  static flags = {
    help: flags.help({ char: 'h' }),
    dryRun: flags.boolean({
      char: 'd',
      description: 'Запустить в режиме dryRun, рассказать, что будет делать, но не делать',
      default: false,
    }),
    master: flags.boolean({
      description: 'Запуск на мастере (чтобы захватить изменения последнего ПР)',
      default: false,
    }),
  };

  async run() {
    const {
      flags: { dryRun, master },
    } = this.parse(Publish);

    const { publishPackages, publishVersions, all } = await this.getAffectedPackages(master);
    const filter = publishPackages.map((name) => `--scope ${name}`).join(' ');
    const packageLockFilter = all.map((name) => `--scope ${name}`).join(' ');

    this.login();

    // Коммитим результат проставления версий
    this.vcs.commitChangedFiles(
      `${ENV.PR_NUMBER ? '#' + ENV.PR_NUMBER : ''} ${publishVersions.join()}`,
    );

    publishPackages.length && this.publishPackages(filter, dryRun);
    !dryRun && this.generateLockFiles(packageLockFilter);

    // Коммитим результат публикации и перегенерированный лок файлов
    this.vcs.commitChangedFiles('lock files');

    if (!dryRun) {
      await this.vcs.forcePush(process.env.BASE_BRANCH);
    }
  }

  async getAffectedPackages(master: boolean) {
    let since: string | undefined = undefined;

    if (master) {
      since = getBaseCommitInTrunk();
    }

    this.log('Считаем измененные пакеты и версии');

    const affected = await new AffectedIver({
      vcs: {
        since,
        autopublishMask: process.env.TOOLCHAIN_AUTOCOMMIT_EXIST ? ENV.AFFECTED : undefined,
      },
    }).execute();

    this.log(JSON.stringify(affected.sortedAffectedPackages));

    this.log('Устанавливаем зависимости для публикации затронутых пакетов');
    const packageNames = affected.sortedAffectedPackages.map(
      (_package) => `--scope ${_package.name}`,
    );
    const filter = packageNames.join(' ');

    if (affected.sortedAffectedPackages.length > 0) {
      exec(`npx lerna bootstrap ${filter}`);
    }

    const versions = await new Version(affected).getVersions();

    const publishVersions: string[] = [];
    const publishPackages: string[] = [];

    versions.forEach((packageChanges, packageName) => {
      switch (packageChanges.severity) {
        case SeverityEnum.NONE:
        case SeverityEnum.UNKNOWN:
          this.log(packageName, 'Ничего не изменилось', packageChanges.reason);
          break;
        default:
          publishVersions.push(`${packageName}@${packageChanges.version}`);
          publishPackages.push(packageName);
          this.log(packageName, 'Новая версия', packageChanges.version);
      }
    });

    return {
      publishVersions,
      publishPackages,
      all: Array.from(versions.keys()),
    };
  }

  login() {
    this.log('npm login');
    exec(
      `${__dirname}/npmlogin ${ENV.ROBOT.NAME} ${ENV.ROBOT.NPM_PASS} ${ENV.ROBOT.EMAIL}`,
      undefined,
      undefined,
      true,
    );
    exec('npm whoami', undefined, undefined, true);

    this.vcs.login();
  }

  generateLockFiles(filter: string) {
    this.log('Генерируем лок файлы');
    exec(`npx lerna exec ${filter} -- npm i --package-lock-only --ignore-scripts`);
  }

  publishPackages(filter: string, dryRun: boolean) {
    this.vcs.reset();

    if (!dryRun) {
      this.log('Публикуем пакеты в npm');
      exec('npx lerna publish from-package --yes --concurrency 4', undefined, undefined, true);
    } else {
      this.log('Проверяем публикацию в npm');
      exec(`npx lerna exec ${filter} -- npm publish --dry-run --concurrency 4`);
    }
  }
}
