import { mkdirSync } from 'fs';
import { join, resolve } from 'path';
import { PerformanceObserver, performance } from 'perf_hooks';

import type { Command } from '@oclif/command';
import { FAILURE_BEHAVIOUR, exec, getBaseCommitInTrunk } from '@yandex-int/frontend.ci.utils';
import { Lerna } from '@yandex-lego/iver';

import { ENV } from '../env';
import { ToolchainVCS } from './interface';

type ArcStatus = {
  changed?: {
    status: string;
    path: string;
  }[];
};

export class Arc implements ToolchainVCS {
  private log: Command['log'];
  private filesToCommit: string[] = [];
  private absoluteMonorepoRoot: string;
  private relativeMonorepoPath: string;
  private branchCommit: string;

  constructor(logger: Command['log']) {
    this.log = logger;
    this.absoluteMonorepoRoot = new Lerna().getRoot();

    // В CI арк монтируется в папку arc_mount
    const arcadiaDir = process.env.MONOREPO_TOOLCHAIN_ARC || 'arc_mount';

    this.relativeMonorepoPath = this.absoluteMonorepoRoot.substring(
      this.absoluteMonorepoRoot.indexOf(arcadiaDir) + arcadiaDir.length,
    );

    this.branchCommit = exec('arc log --oneline -n1 ./ | tail -n1 | cut -c 1-40');
  }

  commitChangedFiles(info: string) {
    const arcStatus = exec(`arc status --json -u no`);
    const parsedArcStatus = JSON.parse(arcStatus);
    const changedFiles = (parsedArcStatus.status as ArcStatus).changed?.map((c) => c.path);

    if (changedFiles) {
      this.log(`${changedFiles}`);
      exec('arc add .');
      exec(`arc commit --no-verify -m '${ENV.AUTOCOMMIT}' -m '${info}' `);
      exec('arc show HEAD', undefined, undefined, true);
      this.filesToCommit.push(...changedFiles);
    }
  }

  login() {
    // Авторизовываемся по токену в TRENBOX_CI задаче
  }

  reset() {
    exec('arc checkout .');
  }

  async forcePush() {
    performance.mark('start');
    const performanceObserver = new PerformanceObserver((items, observer) => {
      const entry = items.getEntriesByName('SVN checkout').pop();
      const time = entry && entry.duration ? entry.duration / 1000 : 0;

      this.log(`${entry?.name}: ${time}s`);
      observer.disconnect();
    });

    performanceObserver.observe({ entryTypes: ['measure'] });

    const svnDestPath = resolve('svn_copy');

    mkdirSync(svnDestPath);

    this.log(`⌛️ Start svn checkout to ${svnDestPath}...`);

    exec(
      `ya tool svn co svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia${this.relativeMonorepoPath} --quiet`,
      FAILURE_BEHAVIOUR.THROW,
      svnDestPath,
    );

    performance.mark('finish');
    performance.measure('SVN checkout', 'start', 'finish');
    this.log(`✅ Checkouted svn copy to ${svnDestPath}`);

    const arcRoot = exec('arc root');

    const folder = this.relativeMonorepoPath.split('/').pop();

    // После чекаута svn, в папке svn_copy появится папка с именем, как в переменной folder.
    // Поэтому знание названия папки нам нужно в дальнейшем, чтобы верно копировать файлы относительно монорепы
    if (!folder) {
      throw new Error(
        `NOT FOUND MONOREPO ROOT ${this.absoluteMonorepoRoot} ${this.relativeMonorepoPath}`,
      );
    }

    for (const file of this.filesToCommit) {
      const absoluteFilePath = resolve(arcRoot, file);
      const relativeFilePath = absoluteFilePath.replace(this.absoluteMonorepoRoot, folder);

      exec(
        `cp ${absoluteFilePath} ${join(svnDestPath, relativeFilePath)}`,
        FAILURE_BEHAVIOUR.THROW,
      );
    }
    // Add names?..
    // SKIP_CHECK - игнор обязательных прекоммитных проверок
    // SKIP_REVIEW - обойти проверку ревью
    const svnCommitMessage = exec(
      `ya tool svn commit -m "${ENV.AUTOCOMMIT};
            SKIP_CHECK
            SKIP_REVIEW"`,
      FAILURE_BEHAVIOUR.THROW,
      join(svnDestPath, folder),
    );

    this.log(`✅ Commited to svn: ${svnCommitMessage}`);
  }

  resetToTrunk() {
    exec(`arc reset --soft ${getBaseCommitInTrunk()}`);
  }

  returnToBrach() {
    exec(`arc checkout ${this.branchCommit}`);
  }
}
