const { Ssh } = require('../lib/ssh');

class SshPortForwarding {
  constructor({ name = 'ssh', deps = [], options }) {
    this.name = name;
    this.deps = deps;
    this.options = options;

    this.ssh = new Ssh({ host: this.options.host });
    this._connect = this._connect.bind(this);
    this._timer = null;
    this._disconnected = false;

    this._logName = `${this.name} - ${this.options.host}`;
  }

  _log(...messages) {
    log(this.name, this.options.host, ...messages);
  }

  _error(...messages) {
    error(this.name, this.options.host, ...messages);
  }

  async _connect(initial = false) {
    const { forwardingPorts, checkInterval = 30 } = this.options;
    const connected = await this.ssh.check();

    let failed = false;
    if (!connected) {
      this._log(`${initial ? 'Connecting' : 'Reconnecting'} to the host`);

      const { code: connectCode, stderr } = await this.ssh.connect();

      if (connectCode === 0) {
        this._log(`The connection to the host has been successfully established`);
      } else {
        failed = true;
        this._error(
          `An error occurred while connecting to the host (exited with code ${connectCode})`,
        );
        this._error(stderr);
      }
    } else if (initial) {
      this._log('The connection already exists');
    }

    if (!failed && (!connected || initial)) {
      const { code: forwardCode, stderr } = await this.ssh.forward(forwardingPorts);

      if (forwardCode === 0) {
        this._log('Ports have been forwarding:');

        forwardingPorts.forEach(({ port, host, hostPort }) => {
          this._log(` - ${port}:${host}:${hostPort}`);
        });
      } else {
        this._error(`An error occurred while portforwarding (exited with code ${forwardCode})`);
        this._error(stderr);
      }
    }

    if (checkInterval > 0) {
      this._timer = setTimeout(this._connect, checkInterval * 1000);
    }
  }

  async _disconnect() {
    if (this._disconnected) {
      return;
    }

    this._disconnected = true;
    clearTimeout(this._timer);
    await this.ssh.disconnect();

    this._log('Closed connection to the host');
  }

  async init() {
    await this._connect(true);
  }

  async stop() {
    await this._disconnect();
  }

  async shutdown() {
    await this._disconnect();
  }

  async terminate() {
    await this._disconnect();
  }
}

const isColorAllowed = () => !process.env.NO_COLOR;
const colorIfAllowed = (colorFn) => (text) => isColorAllowed() ? colorFn(text) : text;
const clc = {
  green: colorIfAllowed((text) => `\x1B[32m${text}\x1B[39m`),
  yellow: colorIfAllowed((text) => `\x1B[33m${text}\x1B[39m`),
  red: colorIfAllowed((text) => `\x1B[31m${text}\x1B[39m`),
  magentaBright: colorIfAllowed((text) => `\x1B[95m${text}\x1B[39m`),
  cyanBright: colorIfAllowed((text) => `\x1B[96m${text}\x1B[39m`),
};

function log(name, host, ...messages) {
  console.log(
    `${clc.green(`[${name}]`)} - ${clc.cyanBright('LOG')}\t${clc.yellow(`[${host}]`)}`,
    ...messages,
  );
}

function error(name, host, ...messages) {
  console.log(
    `${clc.green(`[${name}]`)} - ${clc.red('ERROR')}\t${clc.yellow(`[${host}]`)}`,
    ...messages,
  );
}

module.exports = { SshPortForwarding };
